/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api;

import java.util.List;
import java.util.Set;

import net.nuage.vsp.acs.client.api.model.VspDhcpDomainOption;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspNic;
import net.nuage.vsp.acs.client.api.model.VspStaticNat;
import net.nuage.vsp.acs.client.api.model.VspVm;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.DhcpOption;
import net.nuage.vsp.acs.client.common.model.DhcpOptions;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspException;

public interface NuageVspApiClient {

    List<NuageVspObject> createVMInVSP(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, NetworkDetails attachedNetworkDetails, DhcpOptions dhcpOptions) throws NuageVspException;

    List<NuageVspObject> addVMInterfaceToVM(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, NetworkDetails attachedNetworkDetails,
        NuageVspObject vm, DhcpOptions dhcpOptions) throws NuageVspException;

    // Return Pair to know whether it was created or retrieved
    Pair<Boolean, String> findOrCreateSharedResource(boolean findByExternalId, NuageVspConstants.SharedNetworkType resourceType, String staticNatNetworkUuid,
            String staticNatVlanGateway,
            String staticNatVlanNetmask, boolean staticNatVlanUnderlay, String networkName) throws NuageVspException;

    void deleteSharedResourceInVSP(NuageVspConstants.SharedNetworkType resourceType, String staticNatNetworkUuid, String staticNatVlanGateway, String staticNatVlanNetmask) throws NuageVspException;

    String applyStaticNatInVsp(NetworkDetails attachedNetworkDetails, String vportId, VspStaticNat vspStaticNat) throws NuageVspException;

    enum ExistingDhcpOptionStrategy {
        FETCH,
        FETCH_AND_DELETE,
        IGNORE;

        public boolean isFetch() {
            return this == FETCH || this == FETCH_AND_DELETE;
        }

        public boolean isDeleteUnspecified() {
            return this == FETCH_AND_DELETE;
        }
    }

    /**
     * Login to the Nuage VSP
     * @throws NuageVspException
     */
    void login() throws NuageVspApiException;

    /**
     * Check if an entity exists in the VSD
     * @param entityType The type of the entity
     * @param entityUuid The UUID of the entity
     * @return True if exists, false if not
     */
    boolean entityExists(NuageVspEntity entityType, String entityUuid) throws NuageVspException;

    List<NuageVspObject> buildVmInterfacesList(VspVm vspVm, VspNic... vspNics);

    void deleteVmInterface(String vmUuid, String macAddr, String vmInterfaceID) throws NuageVspException;

    void deleteVM(String vmUuid, String vmId);

    NuageVspObject getVMDetails(String vmUuid) throws NuageVspException;

    String getOrCreateVSPEnterprise(String domainUuid, String domainName, String domainPath) throws NuageVspException;

    Pair<String, String> getOrCreateVSPEnterpriseAndGroup(String networksDomainName, String networksDomainPath, String networksDomainUuid, String networksAccountName,
            String networksAccountUuid) throws NuageVspException;

    String createEnterpriseInVSP(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException;

    void deleteEnterpriseInVsp(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException;

    String createUserInEnterprise(String vsdEnterpriseId, String userNameUuid, String firstName, String lastName, String email, String password) throws NuageVspException;

    NuageVspObject createGroupInEnterprise(String projectOrAccountName, String projectOrAccountUuid, String vsdEnterpriseId) throws NuageVspException;

    Pair<String, String> createNetworkConfigurationWithDefaultACLs(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) throws NuageVspException;

    /**
     * Depending on the option strategy new dhcp options get created on the vsd for a subnet
     * @param existingOptionStrategy FETCH, FETCH_AND_DELETE, IGNORE
     * @param attachedNetworkDetails
     * @param vspNetwork
     * @param dhcpOptions
     * @throws NuageVspException
     */
    void createDhcpOptions(ExistingDhcpOptionStrategy existingOptionStrategy, NetworkDetails attachedNetworkDetails, VspNetwork vspNetwork, DhcpOptions dhcpOptions) throws NuageVspException;

    /**
     * Depending on the option strategy new dhcp options get created on the vsd for the specified vsd entity (vport or subnet)
     * @param existingOptionStrategy
     * @param nuageVspEntity
     * @param nuageVspEntityUuid
     * @param nuageVspEntityId
     * @param vspNetwork
     * @param dhcpOptions
     * @throws NuageVspException
     */
    void createDhcpOptions(ExistingDhcpOptionStrategy existingOptionStrategy, NuageVspEntity nuageVspEntity, String nuageVspEntityUuid, String nuageVspEntityId, VspNetwork vspNetwork,
                           DhcpOptions dhcpOptions) throws NuageVspException;

    /**
     * Removes all specified dhcp options from the specified nic.
     * @param nuageVspEntity
     * @param externalUuid
     * @param entityId
     * @param dhcpOptionCodesToRemove
     * @throws NuageVspException
     */
    void removeAllDhcpOptionsWithCode(NuageVspEntity nuageVspEntity, String externalUuid, String entityId, Set<Integer> dhcpOptionCodesToRemove) throws NuageVspException;

    String createIsolatedL2NetworkWithDefaultACLs(VspNetwork vspNetwork) throws NuageVspException;

    String getEnterprise(String domainUuid) throws NuageVspException;

    Pair<String, String> getIsolatedSubNetwork(String enterpriseId, VspNetwork vspNetwork) throws NuageVspException;

    Pair<String, String> getIsolatedSubNetwork(boolean throwExceptionWhenNotFound, String enterpriseId, VspNetwork vspNetwork) throws NuageVspException;

    String getIsolatedDomain(String enterpriseId, NuageVspEntity attachedNetworkType, String vpcOrSubnetUuid) throws NuageVspException;

    String getIsolatedDomain(boolean throwExceptionWhenNotFound, String enterpriseId, NuageVspEntity attachedNetworkType, String vpcOrSubnetUuid) throws NuageVspException;

    String allocateFIPToVPortInVsp(NetworkDetails attachedNetworkDetails, String vportId, String vspSharedNetworkId, String staticNatIp, String staticNatIpUuid,
            String nicSecondaryIp4Address, String nicSecondaryIpUuid) throws NuageVspException;

    void updateVPortWithFloatingIPId(String vportId, String floatingIPId) throws NuageVspException;

    void releaseFIPFromVsp(NetworkDetails attachedNetworkDetails, String vportId, String staticNatIpUuid, String nicSecondaryIpUuid) throws NuageVspException;

    String findEntityUsingFilter(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, NuageVspFilter filter) throws NuageVspException;

    String getEnterpriseName(String domainName, String domainPath);

    String getResources(NuageVspEntity entityType) throws NuageVspException;

    <T> T getResources(NuageVspEntity entityType, Class<T> type) throws NuageVspException;

    String findEntityUsingFilter(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, NuageVspAttribute filterAttr, Object filterAttrValue)
            throws NuageVspException;

    String findEntityIdByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId) throws NuageVspException;

    @SuppressWarnings("unchecked") <T> T findFieldValueByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId,
            NuageVspAttribute fieldName) throws NuageVspException;

    String findEntityByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId) throws NuageVspException;

    String getEntityId(String jsonString, NuageVspEntity entityType) throws NuageVspException;

    void cleanUpDomainAndTemplate(String enterpriseId, String networkUuid, String domainTemplateName) throws NuageVspException;

    void createVportInVsp(VspNic vspNic, VspNetwork vspNetwork, List<NuageVspObject> vmInterfaceList, NetworkDetails attachedNetworkDetails, DhcpOptions dhcpOptions) throws NuageVspException;

    String generateCmsIdForNuageVsp(String cmsName) throws NuageVspException;

    boolean removeCmsIdForNuageVsp(String cmsId) throws NuageVspException;

    boolean isKnownCmsIdForNuageVsp(String cmsId) throws NuageVspException;

    List<NuageVspObject> parseJsonString(NuageVspEntity entity, String json) throws NuageVspException;

    <T> T parseJsonString(NuageVspEntity entity, String json, Class<T> type) throws NuageVspException;

    void deleteQuietly(NuageVspEntity entityType, String entityId);

    String getResources(NuageVspEntity enterprise, NuageVspFilter filter) throws NuageVspApiException;

    String getResources(NuageVspObject parent, NuageVspEntity childEntityType)  throws NuageVspApiException;
}
