/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api;

import java.util.List;
import java.util.Map;

import net.nuage.vsp.acs.client.api.model.VspAclRule;
import net.nuage.vsp.acs.client.api.model.VspDhcpDomainOption;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspStaticNat;
import net.nuage.vsp.acs.client.exception.NuageVspException;

public interface NuageVspElementClient extends NuageVspClient {

    /**
     * Implement a network
     * @param vspNetwork The network to be implemented
     * @param dnsServers The list of DNS servers to be configured
     * @param ingressFirewallRules The list of ingress firewall rules to be configured
     * @param egressFirewallRules The list of egress firewall rules to be configured
     * @param floatingIpUuids The list of associated floating IP UUIDs
     * @return True if implemented successfully, false if not
     * @throws NuageVspException
     */
    boolean implement(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOption, List<VspAclRule> ingressFirewallRules, List<VspAclRule> egressFirewallRules,
                      List<String> floatingIpUuids) throws NuageVspException;

    /**
     * Apply static NATs to a network
     * @param vspNetwork The network to apply to
     * @param vspStaticNats The static NATs to apply
     * @throws NuageVspException
     */
    void applyStaticNats(VspNetwork vspNetwork, List<VspStaticNat> vspStaticNats) throws NuageVspException;

    /**
     * Apply ACL rules to a network
     * @param vspAclRuleType The type of ACL rules to be applied
     * @param vspNetwork The network to apply to
     * @param vspAclRules The rules to be applied
     * @param networkReset True if the network is being reset
     * @throws NuageVspException
     */
    void applyAclRules(VspAclRule.ACLType vspAclRuleType, VspNetwork vspNetwork, List<VspAclRule> vspAclRules, boolean networkReset) throws NuageVspException;


    /**
     * Shutdown a network
     * @param vspNetwork The network in shutdown
     * @param vspDhcpOption The dhcp options of the network
     */
    boolean shutdownNetwork(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOption);

    /**
     * Shutdown a VPC
     * @param domainUuid The UUID of the domain
     * @param vpcUuid The UUID of the VPC
     * @param domainTemplateName The name of the preconfigured domain template
     * @throws NuageVspException
     */
    void shutdownVpc(String domainUuid, String vpcUuid, String domainTemplateName, List<String> domainRouterUuids) throws NuageVspException;

    /**
     * Sts the extra DHCP options on the specified nic.
     * @param vspNetwork network were the nic is in
     * @param nicUuid
     * @param extraDhcpOptions maps dhcp code to the dhcp value
     * @throws NuageVspException
     */
    void setDhcpOptionsNic(VspNetwork vspNetwork, String nicUuid, Map<Integer, String> extraDhcpOptions) throws NuageVspException;
}
