/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api;

import java.util.List;
import java.util.Map;

import net.nuage.vsp.acs.client.api.model.VspApiDefaults;
import net.nuage.vsp.acs.client.api.model.VspDomain;
import net.nuage.vsp.acs.client.api.model.VspDomainCleanUp;
import net.nuage.vsp.acs.client.api.model.VspDomainTemplate;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.exception.NuageVspException;

public interface NuageVspManagerClient extends NuageVspClient {

    /**
     * Audit a CMS ID
     * @param nuageVspCmsId The CMS ID to audit
     * @param auditOnly True if audit only, false if modify allowed
     * @return A pair of <Success, the CMS ID>
     * @throws NuageVspException
     */
    Pair<Boolean, String> auditNuageVspCmsId(String nuageVspCmsId, boolean auditOnly) throws NuageVspException;

    /**
     * Register a CMS ID
     * @return The registered CMS ID
     * @throws NuageVspException
     */
    String registerNuageVspCmsId() throws NuageVspException;

    /**
     * Unregister a CMS ID
     * @param nuageVspCmsId The CMS ID to unregister
     * @return True if unregistration was successful, false if not
     * @throws NuageVspException
     */
    boolean unregisterNuageVspCmsId(String nuageVspCmsId) throws NuageVspException;

    /**
     * Check if the API version is supported by the client and the vsd
     * @param version The API version to check
     * @return True if supported, false if not
     */
    boolean isSupportedApiVersion(String version) throws NuageVspException;

    /**
     * Get a map of configured defaults
     * @return The map of configured defaults
     * @throws NuageVspException
     */
    VspApiDefaults getApiDefaults() throws NuageVspException;


    /**
     * Sync a domain
     * @param vspDomain The domain to sync
     * @param add True to add the domain
     * @param remove True to remove the domain
     * @return True if syncing was successful, false if not
     * @throws NuageVspException
     * @deprecated use addDomain/deleteDomain instead
     */
    @Deprecated
    boolean syncDomainWithNuageVsp(VspDomain vspDomain, boolean add, boolean remove) throws NuageVspException;

    void addDomain(VspDomain vspDomain) throws NuageVspException;

    void deleteDomain(VspDomain vspDomain) throws NuageVspException;

    /**
     * Clean up a domain
     * @param vspDomainCleanUp The domain to clean up
     * @return True if clean up successful, false if not
     * @throws NuageVspException
     */
    boolean cleanUpDomain(VspDomainCleanUp vspDomainCleanUp) throws NuageVspException;

    /**
     * Lists all the VSP Domain templates for a domain.
     * @param vspDomain The domain where to retrieve the list
     * @return list of domain templates
     */
    List<VspDomainTemplate> getDomainTemplates(VspDomain vspDomain, String filter) throws NuageVspException;

    /**
     * Get supported VSD versions (list is sorted by version number
     * @return list of all the versions the VSD knows and the status of the support for a the versions.
     * @throws NuageVspException
     */
    List<NuageVspApiVersion> getSupportedVersionList() throws NuageVspException;

    /**
     * Get supported VSD versions (list is sorted by version number
     * @return list of all the versions the VSD knows and the status of the support for a the versions.
     * @throws NuageVspException
     */
    Map<NuageVspApiVersion, NuageVspApiVersion.Status> getSupportedVersions() throws NuageVspException;

}