/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Consumer;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.model.Protocol;
import net.nuage.vsp.acs.client.api.model.VspAclRule;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspStaticRoute;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.Acl;
import net.nuage.vsp.acs.client.common.model.AclRulesDetails;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.utils.Logger;
import net.nuage.vsp.acs.client.common.utils.NetUtils;
import net.nuage.vsp.acs.client.common.utils.UuidUtils;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspException;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import static net.nuage.vsp.acs.client.common.model.Acl.AclTemplatePriorityType;
import static net.nuage.vsp.acs.client.common.model.NuageVspFilter.where;

public class NuageVspAclClientImpl implements NuageVspAclClient {
    private static final Logger s_logger = new Logger(NuageVspAclClientImpl.class);

    public static final Random RANDOM = new Random();
    private final NuageVspRestApi api;

    public NuageVspAclClientImpl(NuageVspRestApi api) {
        this.api = api;
    }

    public NuageVspObject getOnlyACLTemplateAssociatedToDomain(String networkUuid, NuageVspEntity domainType, String domainId, NuageVspEntity aclTemplateType) throws
            NuageVspException {
        return Iterables.getOnlyElement(getACLTemplatesAssociatedToDomain(networkUuid, domainType, domainId, aclTemplateType, AclTemplatePriorityType.NONE, true));
    }

    @Override
    public NuageVspObject findACLTemplate(NuageVspEntity parentType, String parentId, NuageVspEntity aclTemplateType, AclTemplatePriorityType priorityType, Integer priority) throws
            NuageVspApiException {
        String aclTemplates;
        switch(priorityType) {
            case TOP:
            case BOTTOM:
                aclTemplates = api.getResources(parentType, parentId, aclTemplateType, NuageVspAttribute.ACLTEMPLATES_PRIORITY_TYPE, priorityType);
                break;
            default:
                aclTemplates = api.getResources(parentType, parentId, aclTemplateType, NuageVspAttribute.ACLTEMPLATES_PRIORITY, priority);
        }

        if (StringUtils.isNotBlank(aclTemplates)) {
            return api.getOnlyJsonEntity(aclTemplateType, aclTemplates);
        }
        return null;
    }

    @Override
    public NuageVspObject findOrCreateACLTemplate(String vpcOrSubnetUuid, NuageVspEntity parentType, String parentId, NuageVspEntity aclTemplateType,
            AclTemplatePriorityType priorityType, Integer priority) throws NuageVspApiException {
        NuageVspObject aclTemplate = findACLTemplate(parentType, parentId, aclTemplateType, priorityType, priority);
        if (aclTemplate != null) {
            return aclTemplate;
        }

        aclTemplate = api.createNuageVspObject(aclTemplateType);
        aclTemplate.setExternalId(vpcOrSubnetUuid);
        StringBuilder name = new StringBuilder();
        switch(aclTemplateType) {
            case INGRESS_ACLTEMPLATES: name.append("Ingress"); break;
            case EGRESS_ACLTEMPLATES: name.append("Egress"); break;
            case EGRESS_DOMAIN_FIP_ACLTEMPLATES: name.append("Egress Domain FIP"); break;
        }
        name.append(" ACL");
        switch(priorityType) {
            case TOP: name.append(" Top"); break;
            case BOTTOM: name.append(" Bottom"); break;
            //case NONE: name.append(priority); break;
        }

        aclTemplate.set(NuageVspAttribute.NAME, name.toString());
        aclTemplate.set(NuageVspAttribute.ACLTEMPLATES_ACTIVE, true);
        aclTemplate.set(NuageVspAttribute.ACLTEMPLATES_PRIORITY_TYPE, priorityType);
        aclTemplate.set(NuageVspAttribute.ACLTEMPLATES_PRIORITY, priority);
        aclTemplate.set(NuageVspAttribute.ACLTEMPLATES_ALLOW_IP, false);
        aclTemplate.set(NuageVspAttribute.ACLTEMPLATES_ALLOW_NON_IP, false);

        try {
            aclTemplate = api.createResource(parentType, parentId, aclTemplate);
            s_logger.debug("Created Default ACLTemplate for network %s in VSP . Response from VSP is %s.", vpcOrSubnetUuid, aclTemplate);
        } catch (NuageVspException e) {
            throw new NuageVspApiException("Failed to create ACL Template", e);
        }

        return aclTemplate;
    }

    @Override
    public List<NuageVspObject> getACLTemplatesAssociatedToDomain(String networkUuid, NuageVspEntity domainType, String domainId, NuageVspEntity aclTemplateType,
            AclTemplatePriorityType priorityType,
            boolean throwExceptionIfNotPresent) throws NuageVspException {
        String aclTemplates ;

        if (priorityType != null) {
            aclTemplates = api.getResources(domainType, domainId, aclTemplateType, NuageVspAttribute.ACLTEMPLATES_PRIORITY_TYPE,
                    priorityType);
        } else {
            aclTemplates = api.getResources(domainType, domainId, aclTemplateType);
        }
        if (StringUtils.isNotBlank(aclTemplates)) {
            //Get the ACLEntries...
            return api.parseJsonString(aclTemplateType, aclTemplates);
        } else if (throwExceptionIfNotPresent) {
            throw new NuageVspApiException(aclTemplateType + " ACL corresponding to network " + networkUuid
                    + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
        } else {
            return null;
        }
    }

    /**
     * Fetches the ACL Entries inside the given ACL Template, that match the given
     * @param aclNetworkLocationId
     * @param aclTemplateType
     * @param aclTemplateId
     * @return
     * @throws Exception
     */
    @Override
    public List<NuageVspObject> getACLEntriesAssociatedToLocation(String aclNetworkLocationId, NuageVspEntity aclTemplateType, String aclTemplateId) throws NuageVspApiException {
        NuageVspEntity aclEntryType = null;
        if (aclTemplateType == NuageVspEntity.INGRESS_ACLTEMPLATES) {
            aclEntryType = NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES;
        } else if (aclTemplateType == NuageVspEntity.EGRESS_ACLTEMPLATES) {
            aclEntryType = NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES;
        } else if (aclTemplateType == NuageVspEntity.EGRESS_DOMAIN_FIP_ACLTEMPLATES) {
            aclEntryType = NuageVspEntity.EGRESS_DOMAIN_FIP_ACLTEMPLATES_ENTRIES;
        }

        if (aclEntryType == null) {
            return Collections.emptyList();
        }

        String aclTemplateEntries = null;
        if (aclNetworkLocationId != null) {
            aclTemplateEntries = api.getResources(aclTemplateType, aclTemplateId, aclEntryType, NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID,
                    aclNetworkLocationId);
        } else {
            aclTemplateEntries = api
                    .getResources(aclTemplateType, aclTemplateId, aclEntryType,
                            where().field(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE).ne(Acl.AclEntryLocationType.SUBNET));
        }

        if (StringUtils.isNotBlank(aclTemplateEntries)) {
            //Get the ACLEntries...
            return api.parseJsonString(aclEntryType, aclTemplateEntries);
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * Fetches the ACL Entries inside the given ACL Template, that match the given
     * @param aclNetworkLocationId
     * @param aclTemplateType
     * @param aclTemplateId
     */
    @Override
    public Map<String, NuageVspObject> getACLEntriesAssociatedToLocationByExternalId(String aclNetworkLocationId, NuageVspEntity aclTemplateType, String aclTemplateId) throws
            NuageVspApiException {
        List<NuageVspObject> entries = getACLEntriesAssociatedToLocation(aclNetworkLocationId, aclTemplateType, aclTemplateId);

        return groupByExternalId(entries);
    }

    @Override
    public void createDefaultRules(VspNetwork vspNetwork, NuageVspEntity nuageVspEntity, String nuageVspEntityId)
            throws NuageVspApiException {

        NuageVspEntity domainTemplateType;
        String domainTemplateId;

        if (nuageVspEntity == NuageVspEntity.DOMAIN_TEMPLATE || nuageVspEntity == NuageVspEntity.L2DOMAIN_TEMPLATE) {
            domainTemplateType = nuageVspEntity;
            domainTemplateId = nuageVspEntityId;
        } else if (nuageVspEntity == NuageVspEntity.DOMAIN || nuageVspEntity == NuageVspEntity.L2DOMAIN) {
            try {
                NuageVspObject domain = api.getResource(nuageVspEntity, nuageVspEntityId);
                domainTemplateId = domain.get(NuageVspAttribute.TEMPLATE_ID);
                domainTemplateType = (nuageVspEntity == NuageVspEntity.DOMAIN) ? NuageVspEntity.DOMAIN_TEMPLATE : NuageVspEntity.L2DOMAIN_TEMPLATE;

            } catch (NuageVspException e) {
                throw NuageVspRestApi.handleException("Failed to create Default Ingress Rules ", e);
            }

        } else return;

        createDefaultIngressAcls(vspNetwork, domainTemplateType, domainTemplateId);
        createDefaultEgressAcls(vspNetwork, domainTemplateType, domainTemplateId);

    }

    @Override
    public Pair<NuageVspObject, NuageVspObject> findOrCreateAclTemplates(NetworkDetails networkDetails, Integer priority) throws NuageVspApiException {

        String vpcOrSubnetUuid = networkDetails.getDomainUuid();
        NuageVspEntity domainType = networkDetails.getDomainType();
        String domainId = networkDetails.getDomainId();

        NuageVspObject ingressTemplate = findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainId, NuageVspEntity.INGRESS_ACLTEMPLATES, AclTemplatePriorityType.NONE, priority);
        NuageVspObject egressTemplate  = findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainId, NuageVspEntity.EGRESS_ACLTEMPLATES, AclTemplatePriorityType.NONE, priority);

        return Pair.of(ingressTemplate, egressTemplate);
    }

    /**
     * Applies ACL rules
     * @param enterpriseId
     * @param network
     * @param aclRulesDetails
     * @param vspAclRule
     * @return
     * @throws net.nuage.vsp.acs.client.exception.NuageVspApiException
     */
    @Override
    public void saveAclRule(String enterpriseId, VspNetwork network, AclRulesDetails aclRulesDetails, AclProgress state, VspAclRule vspAclRule)
            throws NuageVspApiException {

        boolean isNetworkAcl = aclRulesDetails.isNetworkAcl();
        String vspSubnetId = aclRulesDetails.getAclNetworkLocationId();

        NuageVspFilter filter = where(NuageVspAttribute.EXTERNAL_ID).eq(vspAclRule.getUuid())
                .and(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID).eq(vspSubnetId);

        NuageVspObject newParentEntity, oldParentEntity;
        NuageVspEntity newEntityType, oldEntityType;

        if (vspAclRule.getTrafficType().equals(VspAclRule.ACLTrafficType.Ingress)) {
            newParentEntity = aclRulesDetails.getEgressAclTemplate();
            newEntityType = NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES;
            oldParentEntity = aclRulesDetails.getIngressAclTemplate();
            oldEntityType = NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES;

        } else {
            newParentEntity = aclRulesDetails.getIngressAclTemplate();
            newEntityType = NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES;
            oldParentEntity = aclRulesDetails.getEgressAclTemplate();
            oldEntityType = NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES;
        }

        if (vspAclRule.isAddedNetworkAclRule()) {
            String oldEntryJson = api.getResources(oldParentEntity, oldEntityType, filter);
            if (StringUtils.isNotBlank(oldEntryJson)) {
                NuageVspObject oldEntry = api.getFirstJsonEntity(oldEntityType, oldEntryJson);
                s_logger.debug("ACS %s rule %s is getting added to network %s but an VSP Ingress rule is with same UUID %s already exists"
                        + " in VSP. This means the existing CS Egress rule type has been modified to CS Ingress. So, delete this rule from"
                        + " VSP and create a new Egress rule", vspAclRule.getTrafficType(), vspAclRule, network.getName(), vspAclRule.getUuid());

                api.deleteQuietly(oldEntry);
            }
        }

        //create a new ACL Entry
        //get the destVmIp information in staticNat rule case
        String destVmIp = vspAclRule.getSourceIpAddress();


        //Create ACL rule if it does not exists in VSP if the state is Active or Add
        String entryJson = api.getResources(newParentEntity, newEntityType, filter);

        if (StringUtils.isBlank(entryJson)) {
             //Create a egress ACL in VSP
            if (vspAclRule.getTrafficType().equals(VspAclRule.ACLTrafficType.Ingress)) {
                createEgressACLEntryInVsp(enterpriseId, newParentEntity, vspAclRule,  vspSubnetId, network.getId(),
                        state.successfullyAddedEgressACls);

                if (destVmIp != null) {
                    String debugMsg = "ACS Ingress rule %s is getting added to network %s and it does not exists in VSP. The source IP is %s. " +
                            "Enterprise macro with same source IP network will be either created if its not present in VSP. " +
                            "Then a rule will be created using the macro.";
                    s_logger.debug(debugMsg, vspAclRule, network.getName(), destVmIp);

                } else {
                    String debugMsg = "ACS Ingress rule %s is getting added to network %s and it does not exists in VSP. " +
                            "So, ACL rule is created on the L3 Subnet/L2 Domain with network locationId %s on which the ACL rule is added.";
                    s_logger.debug(debugMsg, vspAclRule, network.getName(), vspSubnetId);
                }
            } else {
                createIngressACLEntryInVsp(isNetworkAcl, enterpriseId, newParentEntity, vspAclRule, vspSubnetId, network.getId(),
                        state.successfullyAddedIngressACls);
                String debugMsg = "ACS Egress rule %s is getting added to network %s and it does not exists in VSP. " +
                        "So, ACL rule is created on the L3 Subnet/L2 Domain with network locationId %s on which the ACL rule is added.";

                s_logger.debug(debugMsg, vspAclRule, network.getName(), vspSubnetId);
            }

        } else if (isNetworkAcl) {
            NuageVspObject entry = api.getFirstJsonEntity(newEntityType, entryJson);
            String entryId = entry.getId();
            int oldPriority = entry.get(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY);

            if (vspAclRule.getTrafficType().equals(VspAclRule.ACLTrafficType.Ingress)) {
                updateEgressACLEntryInVsp(enterpriseId, entryId, entry, vspAclRule, vspSubnetId, network.getId(),
                        oldPriority);
            } else {
                updateIngressACLEntryInVsp(enterpriseId, entryId, entry, vspAclRule, vspSubnetId, network.getId(),
                        oldPriority);
            }
        }
    }

    /**
     * Adds or removes the final egress rule of the network, to perform the default action
     * By Default, all outgoing traffic is allowed
     * When an ACL list contains an egress rule, which allows traffic, the default action becomes deny.
     * In case the network has at least one active allowing egress rule. we will add a drop all at the end.
     *
     * Ref: CLOUD-168
     *
     * @param vspNetwork
     * @param aclRulesDetails
     */
    @Override
    public void createOrDeleteDefaultIngressSubnetBlockAcl(VspNetwork vspNetwork, AclRulesDetails aclRulesDetails) throws NuageVspException {

        String networkName = vspNetwork.getName();
        String networkUuid = vspNetwork.getSubnetExternalId();

        try {
            int priority = (int)(NuageVspConstants.DefaultAcl.SUBNET_BLOCK_ACL_PRIORITY+ vspNetwork.getId());

            //Add the default subnet block acl if it is not added get the IngressACLEntry with default priority
            String defaultIngressAcl = api.getResources(NuageVspEntity.INGRESS_ACLTEMPLATES, aclRulesDetails.getIngressAclTemplateId(), NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES,
                    NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY, priority);
            List<NuageVspObject> ingressAclEntry = api.parseJsonString(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, defaultIngressAcl);
            if (aclRulesDetails.isNeedsBlockRule()) {
                if (ingressAclEntry.isEmpty()) {
                    //Create the ACL with default priority
                    try {
                        ensureDefaultIngressAclEntry(null, aclRulesDetails.getIngressAclTemplate(), false, Protocol.ANY, Acl.AclAction.DROP, priority,
                                NuageVspConstants.DefaultAcl.SUBNET_BLOCK_ACL, Acl.AclEntryLocationType.SUBNET, aclRulesDetails.getAclNetworkLocationId(),
                                Acl.AclEntryNetworkType.ANY, networkName, networkUuid);
                        s_logger.debug("Default ACL to block subnets traffic is added with priority " + priority);
                    } catch (NuageVspApiException e) {
                        if (e.getHttpErrorCode() == 409 && e.getNuageErrorCode() == NuageVspConstants.ErrorCode.DUPLICATE_ACL_PRIORITY) {
                            s_logger.debug("Looks like the ACL Entry with priority " + priority + " already exists. So, it is not re-created");
                        }
                    }
                }
            } else if (!ingressAclEntry.isEmpty()) {
                s_logger.debug("There are no Egress ACLs added to the network " + networkName + ". So, delete default subnet block ACL");
                api.deleteQuietly(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, (String)ingressAclEntry.iterator().next().get(NuageVspAttribute.ID));
            }

        } catch (NuageVspException e) {
            throw NuageVspRestApi.handleException("Failed to %s default Subnet ACL to block traffic for network %s",
                    e, aclRulesDetails.isNeedsBlockRule() ? "create" : "delete", networkName);
        }
    }

    @Override
    public void removeAclRule(VspNetwork vspNetwork, AclRulesDetails aclRulesDetails, VspAclRule vspAclRule) throws NuageVspException {
        //Existing ACL has been removed so delete it
        //figure out all the acls that has to be removed
        NuageVspFilter filter = where(NuageVspAttribute.EXTERNAL_ID).eq(vspAclRule.getUuid())
                .and(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID).eq(aclRulesDetails.getAclNetworkLocationId());

        String networkName = vspNetwork.getName()
                ;
        if (vspAclRule.getTrafficType().equals(VspAclRule.ACLTrafficType.Egress)) {

            String ingressEntry = api.getResources(aclRulesDetails.getIngressAclTemplate(), NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, filter);

            if (StringUtils.isNotBlank(ingressEntry)) {
                String ingressEntryId = api.getEntityId(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, ingressEntry);
                s_logger.debug("ACS ACL rule " + vspAclRule + " associated to network " + networkName + " is in Revoke state. This ACL " + ingressEntryId
                        + " exists in VSP. So, delete it");
                api.deleteQuietly(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, ingressEntryId);
            } else {
                s_logger.debug("ACS ACL rule " + vspAclRule + " associated to network " + networkName
                        + " is in Revoke state. But, this ACL does not exist in VSP. So, it is ignored");
            }
        } else if (vspAclRule.getTrafficType().equals(VspAclRule.ACLTrafficType.Ingress)) {
            String egressEntry = api.getResources(aclRulesDetails.getEgressAclTemplate(), NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES, filter);
            if (StringUtils.isNotBlank(egressEntry)) {
                String egressEntryId = api.getEntityId(NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES, egressEntry);
                api.deleteQuietly(NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES, egressEntryId);
                s_logger.debug("ACS ACL rule " + vspAclRule + " associated to network " + networkName + " is in Revoke state. This ACL " + egressEntryId
                        + " exists in VSP and it is deleted");
            } else {
                s_logger.debug("ACS ACL rule " + vspAclRule + " associated to network " + networkName
                        + " is in Revoke state. But, this ACL does not exist in VSP. So, it is ignored");
            }
        }
    }

    @Override
    public void resetAllAclRulesInTheNetwork(VspNetwork vspNetwork, NetworkDetails networkDetails, AclRulesDetails aclRulesDetails) throws NuageVspException {
        createDefaultRules(vspNetwork, networkDetails.getDomainType(), networkDetails.getDomainId());
        cleanStaleAclsFromVsp(aclRulesDetails);
    }

    @Override
    public void applyFIPAccessControl(NuageVspEntity domainType, String domainId, String networkUuid, String staticNatIpUuid, boolean accessControl) throws NuageVspApiException {
        String fipExternalId = networkUuid + ":" + staticNatIpUuid;

        try {
            NuageVspObject floatingIp = api.getEntityByExternalId(domainType, domainId, NuageVspEntity.FLOATING_IP, fipExternalId);

            if (floatingIp != null) {
                floatingIp.set(NuageVspAttribute.FLOATING_IP_ACCESS_CONTROL, accessControl);
                api.updateResource(floatingIp);
            }

        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to apply FIP Access control", exception);
        }
    }

    @Override
    public void applyStaticRoutes(NetworkDetails networkDetails, Collection<VspStaticRoute> staticRoutes) throws NuageVspApiException {
        try {
            NuageVspObject domain = api.getEntityByExternalId(NuageVspEntity.ENTERPRISE, networkDetails.getEnterpriseId(),
                    networkDetails.getDomainType(), networkDetails.getDomainUuid());

            if (domain == null) {
                throw new NuageVspApiException("Domain not found on VSD");
            }

            for (VspStaticRoute staticRoute : staticRoutes) {
                final Pair<String, String> subnetNetmask = NetUtils.cidrToSubnetNetmask(staticRoute.getCidr());
                Map<NuageVspAttribute, Object> filter = Maps.newEnumMap(NuageVspAttribute.class);
                filter.put(NuageVspAttribute.STATIC_ROUTE_ADDRESS, subnetNetmask.getLeft());
                filter.put(NuageVspAttribute.STATIC_ROUTE_NETMASK, subnetNetmask.getRight());
                NuageVspObject entry = api.getEntityByExternalId(NuageVspEntity.DOMAIN, domain.getId(), NuageVspEntity.STATIC_ROUTE, staticRoute.getUuid());

                if (entry == null) {
                    if (!staticRoute.isRevoke()) {
                        entry = api.createNuageVspObject(NuageVspEntity.STATIC_ROUTE);
                        entry.set(NuageVspAttribute.STATIC_ROUTE_ADDRESS, subnetNetmask.getLeft());
                        entry.set(NuageVspAttribute.STATIC_ROUTE_NETMASK, subnetNetmask.getRight());
                        entry.set(NuageVspAttribute.STATIC_ROUTE_NEXTHOP, staticRoute.getNextHop());
                        entry.setExternalId(staticRoute.getUuid());

                        api.createResource(domain, entry);
                    }
                } else {
                    if (staticRoute.isRevoke()) {
                        api.deleteResource(entry);
                    } else if (!staticRoute.getNextHop().equals(entry.get(NuageVspAttribute.STATIC_ROUTE_NEXTHOP))){
                        entry.set(NuageVspAttribute.STATIC_ROUTE_ADDRESS, subnetNetmask.getLeft());
                        entry.set(NuageVspAttribute.STATIC_ROUTE_NETMASK, subnetNetmask.getRight());
                        entry.set(NuageVspAttribute.STATIC_ROUTE_NEXTHOP, staticRoute.getNextHop());
                        entry.setExternalId(staticRoute.getUuid());
                        api.updateResource(entry);
                    }
                }
            }

        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to apply static routes", exception);
        }
    }

    //------------------------------------------------------------------------
    // Helper Methods
    //------------------------------------------------------------------------

    /**
     *  Deletes any stale ACL entries
     *  In case of Firewalls, default ACLs are either added or removed based on the default Egress policy
     *  In case of NetworkAcls, all the ACLs are removed if the associated ACL list is empty
     *  Then, in both Firewall and NetworkACL, stale ACLs are removed....
     */
    private List<String> cleanStaleAclsFromVsp(AclRulesDetails aclRulesDetails) throws NuageVspException {
        List<String> cleanedUpVspAclEntries = new LinkedList<>();

        final Consumer<NuageVspObject> deleteQuietly = api::deleteQuietly;
        final Consumer<NuageVspObject> deleteAndRecord = deleteQuietly.andThen(nvo -> cleanedUpVspAclEntries.add(nvo.getId()));

        final String aclNetworkLocationId = aclRulesDetails.getAclNetworkLocationId();
        if(aclRulesDetails.hasIngressRules()) {
            getACLEntriesAssociatedToLocationByExternalId(aclNetworkLocationId,
                                                          NuageVspEntity.EGRESS_ACLTEMPLATES,
                                                          aclRulesDetails.getEgressAclTemplateId())
                    .values()
                    .stream()
                    .filter(aclRulesDetails::isMissingIngressRule)
                    .forEach(deleteAndRecord);
        }

        if(aclRulesDetails.hasEgressRules()) {
            getACLEntriesAssociatedToLocationByExternalId(aclNetworkLocationId,
                                                          NuageVspEntity.INGRESS_ACLTEMPLATES,
                                                          aclRulesDetails.getIngressAclTemplateId())
                    .values()
                    .stream()
                    .filter(aclRulesDetails::isMissingEgressRule)
                    .forEach(deleteAndRecord);
        }

        return cleanedUpVspAclEntries;
    }

    private void createDefaultIngressAcls(VspNetwork vspNetwork, NuageVspEntity domainType, String domainId) throws NuageVspApiException {

        String vpcOrSubnetUuid = vspNetwork.getVpcOrSubnetInfo().getLeft();
        String networkName = vspNetwork.getName();

        try {
            NuageVspObject topTemplate =  findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainId, NuageVspEntity.INGRESS_ACLTEMPLATES, AclTemplatePriorityType.TOP, 0
            );
            NuageVspObject bottomTemplate =  findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainId, NuageVspEntity.INGRESS_ACLTEMPLATES, AclTemplatePriorityType.BOTTOM, 0
            );

            Map<Integer, NuageVspObject> currentTopAclEntries    = getDefaultAclEntries(topTemplate);
            Map<Integer, NuageVspObject> currentBottomAclEntries = getDefaultAclEntries(bottomTemplate);

            if (vspNetwork.isShared()) {
                ensureDefaultIngressAclEntry(currentBottomAclEntries, bottomTemplate, false, Protocol.ANY, Acl.AclAction.FORWARD,
                                            NuageVspConstants.DefaultAcl.ALLOW_ALL_ACL_PRIORITY, NuageVspConstants.DefaultAcl.INGRESS_ALLOW_ALL_ACL,
                                            Acl.AclEntryLocationType.ANY, null, Acl.AclEntryNetworkType.ANY, networkName, vpcOrSubnetUuid);
                return;
            }

            ensureDefaultIngressAclEntry(currentTopAclEntries, topTemplate, false, Protocol.ANY, Acl.AclAction.FORWARD,
                    NuageVspConstants.DefaultAcl.SUBNET_ALLOW_ACL_PRIORITY, NuageVspConstants.DefaultAcl.SUBNET_ALLOW_ACL, Acl.AclEntryLocationType.ANY, null,
                    Acl.AclEntryNetworkType.ENDPOINT_SUBNET, networkName, vpcOrSubnetUuid);

            //networkAclId will null for firewalls and also VPC tier with no ACL list. Then set the default ACL
            if (vspNetwork.isVpc() || vspNetwork.isEgressDefaultPolicy()) {
                ensureDefaultIngressAclEntry(currentBottomAclEntries, bottomTemplate, true, Protocol.TCP, Acl.AclAction.FORWARD, NuageVspConstants.DefaultAcl.TCP_ALLOW_ACL_PRIORITY,
                        NuageVspConstants.DefaultAcl.INGRESS_ALLOW_TCP_ACL, Acl.AclEntryLocationType.ANY, null, Acl.AclEntryNetworkType.ANY, networkName, vpcOrSubnetUuid);
                ensureDefaultIngressAclEntry(currentBottomAclEntries, bottomTemplate, true, Protocol.UDP, Acl.AclAction.FORWARD, NuageVspConstants.DefaultAcl.UDP_ALLOW_ACL_PRIORITY,
                        NuageVspConstants.DefaultAcl.INGRESS_ALLOW_UDP_ACL, Acl.AclEntryLocationType.ANY, null, Acl.AclEntryNetworkType.ANY, networkName, vpcOrSubnetUuid);
                ensureDefaultIngressAclEntry(currentBottomAclEntries, bottomTemplate, false, Protocol.ICMP, Acl.AclAction.FORWARD, NuageVspConstants.DefaultAcl.ICMP_ALLOW_ACL_PRIORITY,
                        NuageVspConstants.DefaultAcl.INGRESS_ALLOW_ICMP_ACL, Acl.AclEntryLocationType.ANY, null, Acl.AclEntryNetworkType.ANY, networkName, vpcOrSubnetUuid);
            } else {
                deleteDefaultAclEntry(currentBottomAclEntries, NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, NuageVspConstants.DefaultAcl.TCP_ALLOW_ACL_PRIORITY);
                deleteDefaultAclEntry(currentBottomAclEntries, NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, NuageVspConstants.DefaultAcl.UDP_ALLOW_ACL_PRIORITY);
                deleteDefaultAclEntry(currentBottomAclEntries, NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, NuageVspConstants.DefaultAcl.ICMP_ALLOW_ACL_PRIORITY);
            }
        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to create default Ingress ACL for network %s",  exception, vpcOrSubnetUuid);
        }
    }

    private void createDefaultEgressAcls(VspNetwork vspNetwork, NuageVspEntity domainType, String domainTemplateId) throws NuageVspApiException {
        boolean isVpc = vspNetwork.isVpc();
        String vpcOrSubnetUuid = vspNetwork.getVpcOrSubnetInfo().getLeft();
        boolean egressDefaultAllowPolicy = vspNetwork.isEgressDefaultPolicy();
        String networkName = vspNetwork.getName();

        try {
            NuageVspObject topTemplate =  findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainTemplateId, NuageVspEntity.EGRESS_ACLTEMPLATES, AclTemplatePriorityType.TOP,
                    0);
            NuageVspObject bottomTemplate =  findOrCreateACLTemplate(vpcOrSubnetUuid, domainType, domainTemplateId, NuageVspEntity.EGRESS_ACLTEMPLATES,
                    AclTemplatePriorityType.BOTTOM, 0);

            Map<Integer, NuageVspObject> currentTopAclEntries    = getDefaultAclEntries(topTemplate);
            Map<Integer, NuageVspObject> currentBottomAclEntries = getDefaultAclEntries(bottomTemplate);

            if (vspNetwork.isShared()) {
                ensureDefaultEgressAclEntry(currentBottomAclEntries, bottomTemplate, false, Protocol.ANY, Acl.AclAction.FORWARD,
                                      NuageVspConstants.DefaultAcl.ALLOW_ALL_ACL_PRIORITY, NuageVspConstants.DefaultAcl.INGRESS_ALLOW_ALL_ACL,
                                      Acl.AclEntryNetworkType.ANY, networkName);
                return;
            }

            // TOP
            //Default Subnet Allow ACL
            ensureDefaultEgressAclEntry(currentTopAclEntries, topTemplate, false, Protocol.ANY, Acl.AclAction.FORWARD,
                    NuageVspConstants.DefaultAcl.SUBNET_ALLOW_ACL_PRIORITY, NuageVspConstants.DefaultAcl.SUBNET_ALLOW_ACL,
                    Acl.AclEntryNetworkType.ENDPOINT_SUBNET, networkName);

            // BOTTOM =======================================
            if (isVpc) {
                ensureDefaultEgressAclEntry(currentBottomAclEntries, bottomTemplate, false, Protocol.ANY, Acl.AclAction.DROP,
                        NuageVspConstants.DefaultAcl.DOMAIN_BLOCK_ACL_PRIORITY, NuageVspConstants.DefaultAcl.DOMAIN_BLOCK_ACL,
                        Acl.AclEntryNetworkType.ENDPOINT_DOMAIN, networkName);
            }

            if (egressDefaultAllowPolicy) {
                //add ACL : ICMP allow non-reflexive
                ensureDefaultEgressAclEntry(currentBottomAclEntries, bottomTemplate, false, Protocol.ICMP, Acl.AclAction.FORWARD,
                        NuageVspConstants.DefaultAcl.ICMP_ALLOW_ACL_PRIORITY, NuageVspConstants.DefaultAcl.INGRESS_ALLOW_ICMP_ACL,
                        Acl.AclEntryNetworkType.ANY, networkName);
            } else {
                deleteDefaultAclEntry(currentBottomAclEntries, NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, NuageVspConstants.DefaultAcl.ICMP_ALLOW_ACL_PRIORITY);
            }
        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to create default Egress ACL for network %s",  exception, vpcOrSubnetUuid);
        }
    }

    private void deleteDefaultAclEntry(Map<Integer, NuageVspObject> currentAclEntries, NuageVspEntity aclType,  int aclPriority) {
        if(!currentAclEntries.containsKey(aclPriority)) {
            return;
        }

        NuageVspObject aclEntry = currentAclEntries.get(aclPriority);

        final String id = aclEntry.get(NuageVspAttribute.ID);

        api.deleteQuietly(aclType, id);
    }

    private void ensureDefaultEgressAclEntry(Map<Integer, NuageVspObject> currentAclEntries, NuageVspObject aclTemplate, boolean isReflexive, Protocol protocol, Acl.AclAction action, int aclPriority, String aclEntryDescription, Acl.AclEntryNetworkType sourceNetwork, String networkName) throws
            NuageVspException {
        if(currentAclEntries.containsKey(aclPriority)) {
            return;
        }

        NuageVspObject egressACLEntryEntity = api.createNuageVspObject(NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ETHER_TYPE, Acl.AclEtherType.IPv4);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ACTION, action);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE, Acl.AclEntryLocationType.ANY);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_TYPE, sourceNetwork);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PROTOCOL, protocol.getProtocolNumber());
        if (protocol.hasPort()) {
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_SOURCE_PORT, NuageVspConstants.STAR);
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DEST_PORT, NuageVspConstants.STAR);
        }
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DSCP, NuageVspConstants.STAR);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_REFLEXIVE, isReflexive);
        egressACLEntryEntity.set(NuageVspAttribute.DESCRIPTION, aclEntryDescription);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY, aclPriority);

        NuageVspObject egressACLEntry = api.createResource(aclTemplate, egressACLEntryEntity);
        currentAclEntries.put(aclPriority, egressACLEntry);

        s_logger.debug("Created Default egressACLTemplateEntry for network " + networkName + " in VSP . Response from VSP is " + egressACLEntry);
    }

    private void ensureDefaultIngressAclEntry(Map<Integer, NuageVspObject> currentAclEntries, NuageVspObject aclTemplate, boolean isReflexive, Protocol protocol, Acl.AclAction action, int aclPriority, String aclEntryDescription, Acl.AclEntryLocationType locationType, String locationId, Acl.AclEntryNetworkType destinationNetwork, String networkName, String networkUuid) throws
            NuageVspException {
        if(currentAclEntries != null && currentAclEntries.containsKey(aclPriority)) {
            return;
        }

        NuageVspObject ingressACLEntryEntity = api.createNuageVspObject(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ETHER_TYPE, Acl.AclEtherType.IPv4);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ACTION, action);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE, locationType);
        if (locationId != null) {
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID, locationId);
        }
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_TYPE, destinationNetwork);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PROTOCOL, protocol.getProtocolNumber());
        if (protocol.hasPort()) {
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_SOURCE_PORT, NuageVspConstants.STAR);
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DEST_PORT, NuageVspConstants.STAR);
        }
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DSCP, NuageVspConstants.STAR);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_REFLEXIVE, isReflexive);
        ingressACLEntryEntity.set(NuageVspAttribute.DESCRIPTION, aclEntryDescription);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY, aclPriority);
        ingressACLEntryEntity.setExternalId("(" + networkUuid + ")");

        NuageVspObject ingressACLEntry = api.createResource(aclTemplate, ingressACLEntryEntity);
        if (currentAclEntries != null) {
            currentAclEntries.put(aclPriority, ingressACLEntry);
        }

        s_logger.debug("Created Default IngressACLTemplateEntry for network " + networkName + " in VSP . Response from VSP is " + ingressACLEntry);
    }

    private String createAclEntry(NuageVspObject aclTemplate, NuageVspEntity aclTemplateEntryType, String vsdEnterpriseId, VspAclRule rule,
            long networkId, String vspSubnetId) throws NuageVspException {
        boolean isEgress = aclTemplate.getEntityType().equals(NuageVspEntity.EGRESS_ACLTEMPLATES);

        final Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = getAclEntryNetworkForCidr(vsdEnterpriseId, rule.getSourceCidrList());

        NuageVspObject aclEntryEntity = isEgress
                ? buildEgressAclEntry(rule, vspSubnetId, networkId, aclEntryNetwork, false, -1)
                : buildIngressAclEntry(rule, vspSubnetId, networkId, aclEntryNetwork, false, -1);

        return createAclEntry(aclTemplate, aclTemplateEntryType, aclEntryEntity);
    }

    private String createAclEntry(NuageVspObject aclTemplate, NuageVspEntity aclTemplateEntryType, NuageVspObject aclEntryEntity) throws
            NuageVspException {
        NuageVspObject aclEntryJson = api.createResource(aclTemplate, aclEntryEntity, NuageVspConstants.ErrorCode.DUPLICATE_ACL_PRIORITY);
        s_logger.debug("Created " + aclTemplateEntryType + " ACL Entry in VSP. Response from VSP is " + aclEntryJson);
        return aclEntryJson.getId();
    }

    private void createEgressACLEntryInVsp(String vsdEnterpriseId, NuageVspObject egressAclTemplate, VspAclRule rule,
                                          String vspSubnetId, long networkId, List<String> successfullyAddedEgressACls) throws NuageVspApiException {

        try {
            String egressAclEntryId = createAclEntry(egressAclTemplate, NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES, vsdEnterpriseId, rule,
                    networkId, vspSubnetId);
            successfullyAddedEgressACls.add(egressAclEntryId);

        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to create Egress ACL Entry for rule %s in VSP enterprise %s)", exception, rule, vsdEnterpriseId);
        }
    }

    private void createIngressACLEntryInVsp(boolean isNetworkAcl, String vsdEnterpriseId, NuageVspObject ingressAclTemplate, VspAclRule rule,
            String aclNetworkLocationId, long networkId, List<String> successfullyAddedIngressACls) throws NuageVspApiException {

        final Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = getAclEntryNetworkForCidr(vsdEnterpriseId, rule.getSourceCidrList());

        try {
            String ingressAclEntryId = createAclEntry(ingressAclTemplate, NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES, vsdEnterpriseId, rule, networkId,
                    aclNetworkLocationId);
            successfullyAddedIngressACls.add(ingressAclEntryId);
        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to create Ingress ACL Entry for rule %s with CIDR %s in VSP enterprise %s. ", exception, rule, rule.getSourceCidrList(), vsdEnterpriseId);
        }
    }

    private boolean isModified(NuageVspObject origRule, NuageVspObject newRule) {
        return !origRule.entriesEqualTo(newRule);
    }

    private void updateIngressACLEntryInVsp(String vsdEnterpriseId, String ingressAclEntryId, NuageVspObject ingressEntryData, VspAclRule rule,
            String aclNetworkLocationId, long networkId, int oldPriority) throws NuageVspApiException {

        final Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = getAclEntryNetworkForCidr(vsdEnterpriseId, rule.getSourceCidrList());

        try {
            //Check if the ACL is modified or not. If yes, then execute the update method
            NuageVspObject modifiedIngressACLEntryEntity = buildIngressAclEntryByCidr(vsdEnterpriseId, rule, aclNetworkLocationId, networkId, rule.getSourceCidrList(), true,
                    oldPriority);
            if (isModified(modifiedIngressACLEntryEntity, ingressEntryData)) {
                modifiedIngressACLEntryEntity.setId(ingressEntryData.getId());
                //modify Ingress ACL Entry
                String ingressAclEntryJson = api.updateResource(modifiedIngressACLEntryEntity);
                s_logger.debug("Updated Ingress ACL Entry for rule " + rule + " with CIDR " + rule.getSourceCidrList() + " in VSP. Response from VSP is " + ingressAclEntryJson);
            }
        } catch (NuageVspException exception) {
            if (!exception.isNoChangeInEntityException()) {
                throw NuageVspRestApi
                        .handleException("Failed to Modify Ingress ACL Entry for rule %s with CIDR %s in VSP enterprise %s. ", exception, rule, rule.getSourceCidrList(), vsdEnterpriseId);
            }
        }
    }

    private void updateEgressACLEntryInVsp(String vsdEnterpriseId, String egressAclEntryId, NuageVspObject egressEntryData, VspAclRule rule,
            String aclNetworkLocationId, long networkId, int oldPriority) throws NuageVspApiException {

        try {
            //Check if the ACL is modified or not. If yes, then execute the update method
            NuageVspObject modifiedEgressACLEntryEntity = buildEgressAclEntryByCidr(vsdEnterpriseId, rule, aclNetworkLocationId, networkId,
                    true, oldPriority);
            if (isModified(modifiedEgressACLEntryEntity, egressEntryData)) {
                //modify Ingress ACL Entry
                modifiedEgressACLEntryEntity.setId(egressEntryData.getId());
                String ingressAclEntryJson = api.updateResource(modifiedEgressACLEntryEntity);
                s_logger.debug("Updated Ingress ACL Entry for rule %s with CIDR %s in VSP. Response from VSP is %s",
                        rule, rule.getSourceCidrList(), ingressAclEntryJson);
            }
        } catch (NuageVspException exception) {
            if (!exception.isNoChangeInEntityException()) {
                throw NuageVspRestApi.handleException("Failed to Modify Egress ACL Entry for rule %s with CIDR %s in VSP enterprise %s. ",
                        exception, rule, rule.getSourceCidrList(), vsdEnterpriseId);
            }
        }
    }


    private NuageVspObject buildEgressAclEntryByCidr(String vsdEnterpriseId, VspAclRule rule, String aclNetworkLocationId, long networkId,
                                                                 boolean isUpdate, int oldPriority) throws NuageVspApiException {

        Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = null;
        if (rule.getType().equals(VspAclRule.ACLType.NetworkACL)) {
            aclEntryNetwork = getAclEntryNetworkForCidr(vsdEnterpriseId, rule.getSourceCidrList());
        }

        return buildEgressAclEntry(rule, aclNetworkLocationId, networkId, aclEntryNetwork, isUpdate, oldPriority);
    }

    private NuageVspObject buildIngressAclEntryByCidr(String vsdEnterpriseId, VspAclRule rule, String aclNetworkLocationId, long networkId,
                                                                  List<String> sourceCidrList, boolean isUpdate, int oldPriority) throws NuageVspApiException {

        Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = null;

        if (rule.getType().equals(VspAclRule.ACLType.NetworkACL)) {
            aclEntryNetwork = getAclEntryNetworkForCidr(vsdEnterpriseId, sourceCidrList);
        }

        return buildIngressAclEntry(rule, aclNetworkLocationId, networkId,
                aclEntryNetwork, isUpdate, oldPriority);
    }

    private NuageVspObject buildIngressAclEntryBySubnet(VspAclRule rule, String aclNetworkLocationId, long networkId,
            String vsdSubnetId, boolean isUpdate, int oldPriority) throws NuageVspApiException {

        Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork = Pair.of(Acl.AclEntryNetworkType.SUBNET, vsdSubnetId);
        return buildIngressAclEntry(rule, aclNetworkLocationId, networkId, aclEntryNetwork, isUpdate, oldPriority);
    }

    private NuageVspObject buildEgressAclEntry(VspAclRule rule, String aclNetworkLocationId, long networkId,
            Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork,
            boolean isUpdate, int oldPriority) throws NuageVspApiException {
        NuageVspObject egressACLEntryEntity = api.createNuageVspObject(NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ETHER_TYPE, Acl.AclEtherType.IPv4);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ACTION,
                rule.getAction().equals(VspAclRule.ACLAction.Allow) ? Acl.AclAction.FORWARD : Acl.AclAction.DROP);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PROTOCOL, rule.getProtocol().getProtocolNumber());
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DSCP, NuageVspConstants.STAR);
        egressACLEntryEntity.setExternalId(rule.getUuid());

        rule = updatePriorityForAcl(rule, networkId, isUpdate, oldPriority);

        if (rule.getPriority() >= 0 && (rule.getPriority() < NuageVspConstants.MAX_ACL_PRIORITY)) {
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY, rule.getPriority());
        } else {
            String error = "Rule number " + rule.getPriority() + " can not be greater than " + NuageVspConstants.MAX_ACL_PRIORITY + " as it is used as rule numbers for"
                    + " predefined rules in VSP";
            s_logger.error(error);
            throw new NuageVspApiException(error);
        }

        if (rule.getSourceIpAddress() != null) {
            //Address override with the NAT IP which is the source address in the StaticNatRule
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ADDR_OVERRIDE, rule.getSourceIpAddress());
        }
        //If the CS traffic type is ingress rule, when creating a egress rule source port becomes the destination port in VSD
        //Setting the Location and network in ACL
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE, Acl.AclEntryLocationType.SUBNET);
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID, aclNetworkLocationId);

        if (aclEntryNetwork != null) {
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_TYPE, aclEntryNetwork.getLeft());
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_ID, aclEntryNetwork.getRight());
        }

        if (rule.getProtocol().hasPort()) {
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DEST_PORT, rule.getPortRange());
            egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_SOURCE_PORT, NuageVspConstants.STAR);
        }
        egressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_REFLEXIVE, rule.isReflexive());

        return egressACLEntryEntity;
    }

    private NuageVspObject buildIngressAclEntry(VspAclRule rule, String aclNetworkLocationId, long networkId,
            Pair<Acl.AclEntryNetworkType, String> aclEntryNetwork, boolean isUpdate, int oldPriority) throws NuageVspApiException {
        NuageVspObject ingressACLEntryEntity = api.createNuageVspObject(NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ETHER_TYPE, Acl.AclEtherType.IPv4);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ACTION,
                rule.getAction().equals(VspAclRule.ACLAction.Allow) ? Acl.AclAction.FORWARD : Acl.AclAction.DROP);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PROTOCOL, rule.getProtocol().getProtocolNumber());
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DSCP, NuageVspConstants.STAR);
        ingressACLEntryEntity.setExternalId(rule.getUuid());

        //check the priority number for NetworkACL
        rule = updatePriorityForAcl(rule, networkId, isUpdate, oldPriority);

        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY, rule.getPriority());

        if (rule.getType().equals(VspAclRule.ACLType.Firewall)) {
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_ADDR_OVERRIDE, Joiner.on(",").join(rule.getSourceCidrList()));
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_TYPE, Acl.AclEntryNetworkType.ANY);
        } else if (aclEntryNetwork != null) {
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_TYPE, aclEntryNetwork.getLeft());
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_NETWORK_ID,  aclEntryNetwork.getRight());
        }

        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE, Acl.AclEntryLocationType.SUBNET);
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID, aclNetworkLocationId);

        if (rule.getProtocol().hasPort()) {
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_DEST_PORT, rule.getPortRange());
            ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_SOURCE_PORT, NuageVspConstants.STAR);
        }
        ingressACLEntryEntity.set(NuageVspAttribute.ACLTEMPLATES_ENTRY_REFLEXIVE, rule.isReflexive());
        return ingressACLEntryEntity;
    }

    private VspAclRule updatePriorityForAcl(VspAclRule rule, long networkId, boolean isUpdate, int oldPriority) throws NuageVspApiException {
        int newPriority;
        switch(rule.getType()) {
            case NetworkACL:
                if (rule.getPriority() > NuageVspConstants.MIN_ACL_PRIORITY) {
                    String error = "Rule number can not be greater than " + NuageVspConstants.MIN_ACL_PRIORITY + " as it is used to generate a unique rule per tier in VSP";
                    s_logger.error(error);
                    throw new NuageVspApiException(error);
                } else {
                    //In case of NetworkACL, same rule list can be set to different Tiers. So, modify the priority by prepending the tier ID
                    //to make it unique across the tiers under the same VPC
                    newPriority = Integer.valueOf(String.valueOf(networkId) + String.valueOf(rule.getPriority()));
                }
                break;
            default:
                //Set the priority for FW ACL handle the priority for networkACL later
                if (isUpdate) {
                    newPriority = oldPriority;
                } else {
                    newPriority = getRandomPriority();
                }
        }
        return new VspAclRule.Builder().fromObject(rule).priority(newPriority).build();
    }

    private int getRandomPriority() {
        return ((int)(RANDOM.nextDouble() * 99999) % 1000000) + 1;
    }

    private Pair<Acl.AclEntryNetworkType, String> getAclEntryNetworkForCidr(String vsdEnterpriseId, List<String> sourceCidrList) throws NuageVspApiException {
        if (FluentIterable.from(sourceCidrList).anyMatch(new Predicate<String>() {
            @Override public boolean apply(String cidr) {
                return cidr.endsWith("/0");
            }
        })) {
            return  Pair.of(Acl.AclEntryNetworkType.ANY, null);
        } else if (sourceCidrList.size() > 1) {
            if (api.vspHost.getApiVersion() == NuageVspApiVersion.V3_2) {
                throw new NuageVspApiException("Multiple CIDRs are not supported in VSP 3.2");
            }
            String groupId = findOrCreateNetworkMacroGroup(vsdEnterpriseId, Joiner.on(", ").join(sourceCidrList));
            return Pair.of(Acl.AclEntryNetworkType.NETWORK_MACRO_GROUP, groupId);
        } else {
            String macroId = findOrCreateNetworkMacro(vsdEnterpriseId, sourceCidrList.get(0));
            return Pair.of(Acl.AclEntryNetworkType.ENTERPRISE_NETWORK, macroId);
        }
    }

    public String findOrCreateNetworkMacroGroup(String vsdEnterpriseId, String sourceCidrList)
            throws NuageVspApiException {
        try {
            Set<String> macroIds = new TreeSet<String>();
            for (String sourceCidr : sourceCidrList.split(",\\s*")) {
                String macroId = findOrCreateNetworkMacro(vsdEnterpriseId, sourceCidr);
                macroIds.add(macroId);
            }

            String networkMacroGroupUuid = UuidUtils.generateUuidFromCidr(sourceCidrList, null);

            //check to find is the Public Macro exists in enterprise
            NuageVspObject group = api.getEntityByExternalId(
                    NuageVspEntity.ENTERPRISE, vsdEnterpriseId,
                    NuageVspEntity.ENTERPRISE_NTWK_MACRO_GROUP, networkMacroGroupUuid);

            if (group == null) {
                //create a new public network macro with the given netmask and address
                NuageVspObject macroEntity = api.createNuageVspObject(NuageVspEntity.ENTERPRISE_NTWK_MACRO_GROUP);
                macroEntity.set(NuageVspAttribute.NAME, "CIDR List" + networkMacroGroupUuid);
                macroEntity.set(NuageVspAttribute.DESCRIPTION, sourceCidrList);
                macroEntity.setExternalId(networkMacroGroupUuid);
                group = api.createResource(NuageVspEntity.ENTERPRISE, vsdEnterpriseId, macroEntity);
                api.setRelatedEntities(group, NuageVspEntity.ENTERPRISE_NTWK_MACRO, macroIds);
            }
            //If Macro already exists then just return the macroId
            return group.getId();
        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to read Public network macro Group %s in VSP enterprise %s.  " +
                    "Json response from VSP REST API is  %s", exception, sourceCidrList, vsdEnterpriseId,
                    exception.getMessage());
        }
    }

    public String findOrCreateNetworkMacro(String vsdEnterpriseId, String sourceCidr)
            throws NuageVspApiException {
        try {
            Pair<String, String> subnetmask = NetUtils.cidrToSubnetNetmask(sourceCidr);

            //check to find is the Public Macro exists in enterprise
            String macroJsonString = api.getResources(
                    NuageVspEntity.ENTERPRISE,
                    vsdEnterpriseId,
                    NuageVspEntity.ENTERPRISE_NTWK_MACRO,
                    where().field(NuageVspAttribute.ENTERPRISE_NTWK_MACRO_ADDRESS).eq(subnetmask.getLeft())
                            .and()
                            .field(NuageVspAttribute.ENTERPRISE_NTWK_MACRO_NETMASK).eq(subnetmask.getRight())
            );

            if (StringUtils.isBlank(macroJsonString)) {
                //Create an unique uid based on network and cidr
                String networkMacroUuid = UuidUtils.generateUuidFromCidr(sourceCidr);
                //create a new public network macro with the given netmask and address
                NuageVspObject macroEntity = api.createNuageVspObject(NuageVspEntity.ENTERPRISE_NTWK_MACRO);
                macroEntity.set(NuageVspAttribute.NAME, "CIDR " + sourceCidr.replace('.', ' ').replace("/", " - "));
                macroEntity.set(NuageVspAttribute.ENTERPRISE_NTWK_MACRO_ADDRESS, subnetmask.getLeft());
                macroEntity.set(NuageVspAttribute.ENTERPRISE_NTWK_MACRO_NETMASK, subnetmask.getRight());
                macroEntity.setExternalId(networkMacroUuid);
                NuageVspObject macro = api.createResource(NuageVspEntity.ENTERPRISE, vsdEnterpriseId, macroEntity);
                s_logger.debug("Created Enterprise Network Macro in VSP. Response from VSP is " + macro);
                return macro.getId();
            }

            //If Macro already exists then just return the macroId
            return api.getEntityId(NuageVspEntity.ENTERPRISE_NTWK_MACRO, macroJsonString);
        } catch (NuageVspException exception) {
            throw NuageVspRestApi.handleException("Failed to read Public network macro %s in VSP enterprise %s.  " +
                            "Json response from VSP REST API is  %s", exception, sourceCidr, vsdEnterpriseId,
                    exception.getMessage());
        }
    }

    public Map<String, NuageVspObject> groupByExternalId(List<NuageVspObject> aclEntries) {
        Map<String, NuageVspObject> externalUuidToAcl = new HashMap<>();
        for (NuageVspObject entry : aclEntries) {
            String externalUuid = entry.getExternalId();
            if (externalUuid != null) {
                externalUuidToAcl.put(externalUuid, entry);
            }
        }

        return externalUuidToAcl;
    }

    /**
     * Fetches the current Default ACL Entries
     *
     * @param aclTemplate
     * @return
     * @throws Exception
     */
    public Map<Integer, NuageVspObject> getDefaultAclEntries(NuageVspObject aclTemplate) throws NuageVspException {
        List<NuageVspObject> defaultVspAclEntries = getACLEntriesAssociatedToLocation(null, aclTemplate.getEntityType(), aclTemplate.getId());
        Map<Integer, NuageVspObject> aclPriorityToEntry = new HashMap<Integer, NuageVspObject>();
        for (NuageVspObject aclEntry : defaultVspAclEntries) {
            Integer priority = aclEntry.get(NuageVspAttribute.ACLTEMPLATES_ENTRY_PRIORITY);
            aclPriorityToEntry.put(priority, aclEntry);
        }
        return aclPriorityToEntry;
    }
}