/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.NuageVspApiClient;
import net.nuage.vsp.acs.client.api.NuageVspClient;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.exception.NuageVspException;

public class NuageVspClientImpl implements NuageVspClient {

    protected NuageVspApiClient nuageVspApiClient;
    protected NuageVspAclClient nuageVspAclClient;

    public NuageVspClientImpl(NuageVspApiClient nuageVspApiClient, NuageVspAclClient nuageVspAclClient) {
        this.nuageVspApiClient = nuageVspApiClient;
        this.nuageVspAclClient = nuageVspAclClient;
    }

    @Override
    public NetworkDetails getAttachedNetworkDetails(VspNetwork vspNetwork) throws NuageVspException {
        return getAttachedNetworkDetails(true, vspNetwork);
    }

    @Override
    public NetworkDetails getAttachedNetworkDetails(boolean throwExceptionWhenNotFound, VspNetwork vspNetwork) throws NuageVspException {
        NetworkDetails attachedNetworkDetails = new NetworkDetails();
        String enterpriseId = nuageVspApiClient.getEnterprise(vspNetwork.getVspDomain().getUuid());
        attachedNetworkDetails.setEnterpriseId(enterpriseId);
        Pair<String, String> vsdDomainAndSubnetId;
        if (vspNetwork.isL3() || vspNetwork.isVpc() || vspNetwork.isShared()) {
            attachedNetworkDetails.setIsVpc(vspNetwork.isVpc());
            attachedNetworkDetails.setDomainType(NuageVspEntity.DOMAIN);
            attachedNetworkDetails.setSubnetType(NuageVspEntity.SUBNET);
            if (vspNetwork.isVpc()) {
                attachedNetworkDetails.setDomainUuid(vspNetwork.getVpcUuid());
            } else {
                attachedNetworkDetails.setDomainUuid(vspNetwork.getUuid());
            }

            vsdDomainAndSubnetId = nuageVspApiClient.getIsolatedSubNetwork(throwExceptionWhenNotFound, enterpriseId, vspNetwork);
            if (vsdDomainAndSubnetId == null) {
                return null;
            }

            attachedNetworkDetails.setDomainId(vsdDomainAndSubnetId.getLeft());
            attachedNetworkDetails.setSubnetId(vsdDomainAndSubnetId.getRight());
        } else {
            attachedNetworkDetails.setDomainType(NuageVspEntity.L2DOMAIN);
            attachedNetworkDetails.setSubnetType(NuageVspEntity.L2DOMAIN);
            attachedNetworkDetails.setDomainUuid(vspNetwork.getUuid());

            String domainId = nuageVspApiClient.getIsolatedDomain(throwExceptionWhenNotFound, enterpriseId, NuageVspEntity.L2DOMAIN, vspNetwork.getUuid());
            if (domainId == null) {
                return null;
            }

            attachedNetworkDetails.setSubnetId(domainId);
            attachedNetworkDetails.setDomainId(domainId);
        }
        return attachedNetworkDetails;
    }
}
