/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.NuageVspApiClient;
import net.nuage.vsp.acs.client.api.NuageVspElementClient;
import net.nuage.vsp.acs.client.api.model.VspAclRule;
import net.nuage.vsp.acs.client.api.model.VspAclRule.ACLState;
import net.nuage.vsp.acs.client.api.model.VspAclRule.ACLTrafficType;
import net.nuage.vsp.acs.client.api.model.VspDhcpDomainOption;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspStaticNat;
import net.nuage.vsp.acs.client.common.model.AclRulesDetails;
import net.nuage.vsp.acs.client.common.model.Dhcp;
import net.nuage.vsp.acs.client.common.model.DhcpOption;
import net.nuage.vsp.acs.client.common.model.DhcpOptions;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.utils.Logger;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspException;
import net.nuage.vsp.acs.client.exception.NuageVspUnsupportedRequestException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.apache.commons.lang3.tuple.Pair;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import static net.nuage.vsp.acs.client.api.NuageVspApiClient.ExistingDhcpOptionStrategy;
import static net.nuage.vsp.acs.client.common.model.NuageVspFilter.where;

public class NuageVspElementClientImpl extends NuageVspClientImpl implements NuageVspElementClient {
    private final Logger s_logger = new Logger(NuageVspElementClientImpl.class);

    public NuageVspElementClientImpl(NuageVspApiClient nuageVspApiClient, NuageVspAclClient nuageVspAclClient) {
        super(nuageVspApiClient, nuageVspAclClient);
    }

    @Override
    public boolean implement(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions, List<VspAclRule> ingressFirewallRules, List<VspAclRule> egressFirewallRules,
                             List<String> floatingIpUuids) throws NuageVspException {
        NetworkDetails attachedNetworkDetails;
        try {
            attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        } catch (NuageVspApiException exception) {
            s_logger.error("Exception occurred while executing implement API. So, FIP clean up could not be execued successfully. Retry restarting the network "
                    + vspNetwork.getName());
            return true;
        }

        s_logger.debug("Starting the sync for network " + vspNetwork.getName() + " at " + new Date());

        // apply firewall rules
        if (!vspNetwork.isVpc() && vspNetwork.isFirewallServiceSupported()) {
            s_logger.debug("Started Sync Ingress Firewall Rule for network " + vspNetwork.getName() + " at " + new Date());
            applyAclRules(VspAclRule.ACLType.Firewall, vspNetwork, ingressFirewallRules, false);
            s_logger.debug("Finished Sync Ingress Firewall Rule for network " + vspNetwork.getName() + " at " + new Date());

            s_logger.debug("Started Sync Egress Firewall Rule for network " + vspNetwork.getName() + " at " + new Date());
            applyAclRules(VspAclRule.ACLType.Firewall, vspNetwork, egressFirewallRules, false);
            s_logger.debug("Finished Sync Egress Firewall Rule for network " + vspNetwork.getName() + " at " + new Date());
        }

        s_logger.debug("Started Sync of Static NAT for network " + vspNetwork.getName() + " at " + new Date());
        //This method is called when a Network is restarted with cleanup true or false. Also, when the network is newly created
        //We can handle the logic to clean all the stale Static Ips
        //Get all the NAT public IPs associated with the network
        try {
            updateDhcpOptions(vspNetwork, vspDhcpOptions, attachedNetworkDetails);
            //Get all the Floating Ips associated to this network filtered by the Floating IP's External ID that has networkUUID:publicIPUUID
            //example : externalID beginswith 'networkUuid'
            NuageVspFilter fipExternalIdFilter = where().field(NuageVspAttribute.EXTERNAL_ID).startsWith(attachedNetworkDetails.getDomainUuid());
            String floatingIpsAssoToNetwork = nuageVspApiClient.findEntityUsingFilter(attachedNetworkDetails.getDomainType(), attachedNetworkDetails.getDomainId(),
                    NuageVspEntity.FLOATING_IP, fipExternalIdFilter);
            if (!vspNetwork.isPublicAccess() && StringUtils.isNotBlank(floatingIpsAssoToNetwork)) {
                List<NuageVspObject> fips = nuageVspApiClient.parseJsonString(NuageVspEntity.FLOATING_IP, floatingIpsAssoToNetwork);
                for (NuageVspObject fip : fips) {
                    String fipId = fip.getId();
                    String externalId = fip.getExternalId();
                    String fipIp = fip.get(NuageVspAttribute.FLOATING_IP_ADDRESS);
                    if (!floatingIpUuids.contains(externalId.substring(externalId.indexOf(":") + 1))) {
                        s_logger.debug("Floating IP " + fipIp + " with " + externalId + " does not exists in ACS network " + vspNetwork.getName()
                                + ". So, processing to clean the stale FIP from VSP");
                        //get the all VPorts and check for the FIP associated to it
                        String vportJson = null;
                        try {
                            vportJson = nuageVspApiClient.getResources(fip, NuageVspEntity.VPORT);
                        } catch (NuageVspApiException e) {
                            s_logger.debug("Failed to get VPorts from VSP during FIP clean up. " + e.getMessage());
                        }

                        if (StringUtils.isNotBlank(vportJson)) {
                            List<NuageVspObject> vports = nuageVspApiClient.parseJsonString(NuageVspEntity.VPORT, vportJson);
                            for (NuageVspObject vport : vports) {
                                if (StringUtils.equals(vport.<String>get(NuageVspAttribute.VPORT_FLOATING_IP_ID), fipId)) {
                                    String vportId = vport.get(NuageVspAttribute.ID);
                                    nuageVspApiClient.updateVPortWithFloatingIPId(vportId, null);
                                    s_logger.debug("Found a VPort " + vport + " that is associated the stale FIP in network " + vspNetwork.getName()
                                            + ". Removed the association to clean the FIP " + fipIp);
                                }
                            }
                        }
                        s_logger.debug("Clean the stale FIP " + fipIp + " associated to network " + vspNetwork.getName() + " from VSP");
                        nuageVspApiClient.deleteQuietly(NuageVspEntity.FLOATING_IP, fipId);
                    }
                }
            }
        } catch (NuageVspApiException exception) {
            s_logger.error("Exception occurred while executing implement API. So, FIP clean up could not be execued successfully. Retry restarting the network "
                    + vspNetwork.getName());
            return false;
        }
        return true;
    }

    private boolean usesPreconfiguredDomainTemplate(String enterpriseId, String networkUuid, String domainTemplateName) throws NuageVspException {
        String networkDomainTemplateId = nuageVspApiClient.findFieldValueByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN,
                networkUuid, NuageVspAttribute.TEMPLATE_ID);
        String domainTemplateEntity = nuageVspApiClient.findEntityUsingFilter(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE,
                NuageVspAttribute.NAME, domainTemplateName);
        String domainTemplateId = nuageVspApiClient.getEntityId(domainTemplateEntity, NuageVspEntity.DOMAIN_TEMPLATE);
        return StringUtils.isNotBlank(networkDomainTemplateId) && StringUtils.isNotBlank(domainTemplateId) && networkDomainTemplateId.equals(domainTemplateId);
    }

    public void applyStaticNats(VspNetwork vspNetwork, List<VspStaticNat> vspStaticNats) throws NuageVspException {
        //Below is a hack to figure out the domain or l2 domain to which the floating Ip is attached to
        //This is useful to delete the FLoating IP when we can not get VM information from the StaticNat
        NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        for (VspStaticNat vspStaticNat : vspStaticNats) {
            String vspNicUuid = vspStaticNat.getVspNic() != null ? vspStaticNat.getVspNic().getUuid() : null;
            String vspNicSecondaryIpUuid = vspStaticNat.getVspNic() != null ? vspStaticNat.getVspNic().getSecondaryIpUuid() : null;
            if (vspStaticNat.getVspNic() != null && vspStaticNat.getVspNic().getMacAddress() == null && vspStaticNat.getRevoke() == Boolean.TRUE) {
                //this is case where the VM was deleted without disabling the Static NAT
                //We need to find the Subnet and Domain to which this StaticNat belongs to..
                s_logger.debug("VM is getting deleted with out disabling the Static NAT " + vspStaticNat.getIpAddress() + "(" + vspStaticNat.getIpUuid() + ")"
                        + ". So, floating IP is disassociated from VM and is deleted from VSP");
                nuageVspApiClient.releaseFIPFromVsp(attachedNetworkDetails, null, vspStaticNat.getIpUuid(), vspNicSecondaryIpUuid);
            } else {
                String vportId = nuageVspApiClient.findEntityIdByExternalUuid(attachedNetworkDetails.getDomainType(), attachedNetworkDetails.getDomainId(),
                        NuageVspEntity.VPORT, vspNicUuid);
                if (vspStaticNat.getRevoke() == Boolean.TRUE) {
                    s_logger.debug("Static NAT %s (%s) is deleted from the VM. So, disassociate the Floating IP from VM's VPort %s and delete the Floating IP", vspStaticNat.getIpAddress(), vspStaticNat.getIpUuid(), vportId);
                    nuageVspApiClient.releaseFIPFromVsp(attachedNetworkDetails, vportId, vspStaticNat.getIpUuid(), vspNicSecondaryIpUuid);
                } else {
                    s_logger.debug("Static NAT " + vspStaticNat.getIpAddress() + "(" + vspStaticNat.getIpUuid() + ") is associated to the VM. " +
                            "So, create a new Floating IP " + vspStaticNat.getIpAddress() + " and associate it to VM with VPort " + vportId);
                    nuageVspApiClient.applyStaticNatInVsp(attachedNetworkDetails, vportId, vspStaticNat);
                }
            }
        }
    }

    public void applyAclRules(VspAclRule.ACLType vspAclRuleType, VspNetwork vspNetwork, List<VspAclRule> vspAclRules, boolean networkReset) throws NuageVspException {
        int random = RandomUtils.nextInt(1000);
        long initialStartTime = System.currentTimeMillis();
        Boolean ingressFirewallRules = null;

        if (vspAclRules == null) {
            vspAclRules = Lists.newArrayList();
        }

        if (CollectionUtils.isNotEmpty(vspAclRules) && vspAclRuleType == VspAclRule.ACLType.Firewall) {
            ingressFirewallRules = vspAclRules.get(0).getTrafficType() == ACLTrafficType.Ingress;
        }

        NetworkDetails attachedNetworkDetails;
        try {
            attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        } catch (NuageVspException e) {
            s_logger.debug("Enterprise or Domain does not exists in VSP. So, ACLs update is ignored for network " + vspNetwork.getName() + ". " + e.getMessage());
            return;
        }

        boolean usesPreConfiguredDomainTemplate = usesPreconfiguredDomainTemplate(attachedNetworkDetails.getEnterpriseId(),
                attachedNetworkDetails.getDomainUuid(), vspNetwork.getDomainTemplateName());
        if (usesPreConfiguredDomainTemplate) {
            if (CollectionUtils.isNotEmpty(vspAclRules)) {
                for (VspAclRule rule : vspAclRules) {
                    if (rule.getState() != ACLState.Revoke) {
                        s_logger.info("CloudStack ACLs are not supported with Nuage Preconfigured Domain Template");
                        throw new NuageVspUnsupportedRequestException("CloudStack ACLs are not supported with Nuage Preconfigured Domain Template");
                    }
                }
            }
            return;
        }

        if (StringUtils.isNotBlank(attachedNetworkDetails.getDomainId())) {

            NuageVspObject ingressAclTemplate;
            NuageVspObject egressAclTemplate;

            {
                final Pair<NuageVspObject, NuageVspObject> aclTemplates = nuageVspAclClient.findOrCreateAclTemplates(attachedNetworkDetails, 1);
                ingressAclTemplate = aclTemplates.getLeft();
                egressAclTemplate = aclTemplates.getRight();
            }

            //To set the location field in the ACL get the Subnet Id it is a L3 domain
            String aclNetworkLocationId = attachedNetworkDetails.getSubnetId();
            if (!StringUtils.isBlank(aclNetworkLocationId)) {
                AclRulesDetails aclRulesDetails = new AclRulesDetails(vspAclRules,
                                                                      vspAclRuleType == VspAclRule.ACLType.NetworkACL,
                                                                      ingressFirewallRules,
                                                                      true,
                                                                      aclNetworkLocationId,
                                                                      ingressAclTemplate,
                                                                      egressAclTemplate);

                if (networkReset) {
                    s_logger.debug("Network is restarted to just cleanup the stale ACL rules and checking for default rules for " + vspNetwork.getName());
                    nuageVspAclClient.resetAllAclRulesInTheNetwork(vspNetwork, attachedNetworkDetails, aclRulesDetails);
                    return;
                }

                if (ingressAclTemplate != null && egressAclTemplate != null) {
                    updateACLEntriesInVsp(attachedNetworkDetails, vspNetwork, aclRulesDetails);
                }
                s_logger.debug("Network " + vspNetwork.getName() + "(" + random + ")   total time taken to process this thread with " + vspAclRules.size() + " rules is "
                        + (System.currentTimeMillis() - initialStartTime));
            } else {
                s_logger.debug("VSP subnet corresponding to network " + vspNetwork.getName() + " is not found. So, skipping ACL application");
            }
        }
    }

    private void updateACLEntriesInVsp(NetworkDetails attachedNetworkDetails, VspNetwork vspNetwork,
            AclRulesDetails aclRulesDetails) throws NuageVspException {
        NuageVspAclClient.AclProgress aclProgress = new NuageVspAclClient.AclProgress();

        //This is to figure if the use case is replace ACL

        String enterpriseId = attachedNetworkDetails.getEnterpriseId();

        if (aclRulesDetails.isAllRulesActive()) {
            s_logger.debug("All rules are in in Active state. This is a ACLList replace scenario or Network Restart. So, processing the ACL lists.");
            nuageVspAclClient.resetAllAclRulesInTheNetwork(vspNetwork, attachedNetworkDetails, aclRulesDetails);

            for (VspAclRule vspAclRule : aclRulesDetails.getVspAclRules()) {
                nuageVspAclClient.saveAclRule(enterpriseId, vspNetwork, aclRulesDetails, aclProgress, vspAclRule);
            }
        } else {
            for (VspAclRule vspAclRule : aclRulesDetails.getVspAclRules()) {
                //Convert Firewall Rule and NetworkACLItem to a common ACLRule for easy manipulation
                switch (vspAclRule.getState()) {
                case Add:
                    nuageVspAclClient.saveAclRule(enterpriseId, vspNetwork, aclRulesDetails, aclProgress, vspAclRule);
                    break;
                case Revoke:
                    nuageVspAclClient.removeAclRule(vspNetwork, aclRulesDetails, vspAclRule);
                    break;
                }
            }
        }

        if (aclRulesDetails.isNetworkAcl()) {
            nuageVspAclClient.createOrDeleteDefaultIngressSubnetBlockAcl(vspNetwork, aclRulesDetails);

        }
    }

    @Override
    public boolean shutdownNetwork(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) {
        try {
            NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(false, vspNetwork);
            if (attachedNetworkDetails == null) {
                s_logger.debug("The network details corresponding to network " + vspNetwork.getUuid() + " could not be found.");
                return true;
            }

            updateDhcpOptions(vspNetwork, vspDhcpOptions, attachedNetworkDetails);
        } catch (NuageVspException e) {
            s_logger.error("Exception occurred while executing shutdown network API. name: " + vspNetwork.getName());
            return false;
        }
        return true;
    }

    private void updateDhcpOptions (VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions, NetworkDetails attachedNetworkDetails) throws NuageVspException {

        if (vspDhcpOptions != null && !vspNetwork.isL2()) {
            try {
                s_logger.debug("Started Sync of DNS Server setting for network " + vspNetwork.getName() + " at " + new Date());
                DhcpOptions dhcpOptions;
                if (vspDhcpOptions.getVrIsDnsProvider()) {
                    dhcpOptions = new DhcpOptions(vspNetwork.getVirtualRouterIp(), vspDhcpOptions.getDnsServers(), vspDhcpOptions.getNetworkDomain());
                } else {
                    dhcpOptions = new DhcpOptions(null, vspDhcpOptions.getDnsServers(), null);
                }
                nuageVspApiClient.createDhcpOptions(ExistingDhcpOptionStrategy.FETCH_AND_DELETE, attachedNetworkDetails, vspNetwork, dhcpOptions);
            } catch (NuageVspException e1) {
                s_logger.warn("Failed to update the DNS Server information for network " + vspNetwork.getName());
            }
            s_logger.debug("Finished Sync of DNS Server setting for network " + vspNetwork.getName() + " at " + new Date());
        }
    }

    @Override
    public void shutdownVpc(String domainUuid, String vpcUuid, String domainTemplateName, List<String> domainRouterUuids) throws NuageVspException {
        String enterpriseId = nuageVspApiClient.getEnterprise(domainUuid);
        if (StringUtils.isNotBlank(enterpriseId)) {
            if (CollectionUtils.isNotEmpty(domainRouterUuids)) {
                for (String domainRouterUuid : domainRouterUuids) {
                    NuageVspObject domainRouter = nuageVspApiClient.getVMDetails(domainRouterUuid);
                    if (domainRouter != null) {
                        String vmId = domainRouter.get(NuageVspAttribute.ID);
                        nuageVspApiClient.deleteVM(domainRouterUuid, vmId);
                        s_logger.debug("VR " + domainRouterUuid + " in VPC " + vpcUuid + " is deleted as the VPC is deleted");
                    }
                }
            }

            //get the L3 DomainTemplate with externalUuid
            String domainTemplateId = nuageVspApiClient.findFieldValueByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN,
                    vpcUuid, NuageVspAttribute.TEMPLATE_ID);
            String vpcDomainTemplateEntity = nuageVspApiClient.findEntityUsingFilter(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE,
                    NuageVspAttribute.NAME, domainTemplateName);
            String currentPredefinedDomTplId = nuageVspApiClient.getEntityId(vpcDomainTemplateEntity, NuageVspEntity.DOMAIN_TEMPLATE);

            if (domainTemplateId == null) return;

            //pre-defined domain template - only delete the domain not the DT.
            if (domainTemplateId.equals(currentPredefinedDomTplId)) {
                String vspNetworkId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN, vpcUuid);
                if (StringUtils.isNotBlank(vspNetworkId)) {
                    s_logger.debug("Deleting VPC " + vpcUuid + " from VSP");
                    nuageVspApiClient.deleteQuietly(NuageVspEntity.DOMAIN, vspNetworkId);
                }
            } else { //delete domain template and domain under DT.
                String vspNetworkId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE, vpcUuid);
                if (StringUtils.isNotBlank(vspNetworkId)) {
                    s_logger.debug("Deleting VPC " + vpcUuid + " from VSP");
                    nuageVspApiClient.deleteQuietly(NuageVspEntity.DOMAIN_TEMPLATE, vspNetworkId);
                } else {
                    // Added fallback for scenario described in CLOUD-406 (change default DT)
                    vspNetworkId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN, vpcUuid);
                    if (StringUtils.isNotBlank(vspNetworkId)) {
                        s_logger.debug("Deleting VPC " + vpcUuid + " from VSP");
                        nuageVspApiClient.deleteQuietly(NuageVspEntity.DOMAIN, vspNetworkId);
                    }
                }
            }
        }
    }

    @Override
    public void setDhcpOptionsNic(VspNetwork vspNetwork, String nicUuid, Map<Integer, String> extraDhcpOptions) throws NuageVspException {
        DhcpOptions dhcpOptionsToApply = extractDhcpOptions(extraDhcpOptions);
        NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        String vportVsdId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.SUBNET, attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT, nicUuid);

        nuageVspApiClient.removeAllDhcpOptionsWithCode(NuageVspEntity.VPORT, nicUuid, vportVsdId, dhcpOptionsToApply.getOptionsToBeRemoved());
        nuageVspApiClient.createDhcpOptions(ExistingDhcpOptionStrategy.FETCH, NuageVspEntity.VPORT, nicUuid, vportVsdId, vspNetwork, dhcpOptionsToApply);
    }

    private DhcpOptions extractDhcpOptions(Map<Integer, String> actualDhcpOptions) {
        DhcpOptions toBeAddedDhcpOptions = new DhcpOptions();
        Set<Integer> supportedDhcpCodes = Dhcp.dhcpCodeToType.keySet()
                                                             .stream().map(Dhcp.DhcpOptionCode::getCode)
                                                             .collect(Collectors.toSet());
        Set<Integer> defaultDhcpCodes = Dhcp.defaultDhcpOptions.stream().map(Dhcp.DhcpOptionCode::getCode)
                                                                .collect(Collectors.toSet());


        if(!Sets.difference(actualDhcpOptions.keySet(), supportedDhcpCodes).isEmpty() || !Sets.intersection(actualDhcpOptions.keySet(), defaultDhcpCodes).isEmpty()) {
            throw new IllegalArgumentException("Unsupported DHCP option specified.");
        }

        for(Integer dhcpCode: actualDhcpOptions.keySet()) {
            String dhcpCurrentCodeOption = actualDhcpOptions.get(dhcpCode);

            if(StringUtils.isNotBlank(dhcpCurrentCodeOption)) {
                DhcpOption option = new DhcpOption(dhcpCode, dhcpCurrentCodeOption);
                toBeAddedDhcpOptions.addOption(option);
            } else { //in case the options are set to an empty string or are not specified
                toBeAddedDhcpOptions.addOptionToRemove(dhcpCode);
            }
        }

        return toBeAddedDhcpOptions;
    }
}
