/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.util.List;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.NuageVspApiClient;
import net.nuage.vsp.acs.client.api.NuageVspGuruClient;
import net.nuage.vsp.acs.client.api.model.VspAddressRange;
import net.nuage.vsp.acs.client.api.model.VspDhcpDomainOption;
import net.nuage.vsp.acs.client.api.model.VspDhcpVMOption;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspNic;
import net.nuage.vsp.acs.client.api.model.VspStaticNat;
import net.nuage.vsp.acs.client.api.model.VspVm;
import net.nuage.vsp.acs.client.common.model.DhcpOptions;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.common.utils.Logger;
import net.nuage.vsp.acs.client.common.utils.UuidUtils;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspException;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import static net.nuage.vsp.acs.client.common.NuageVspConstants.SharedNetworkType;

public class NuageVspGuruClientImpl extends NuageVspClientImpl  implements NuageVspGuruClient {
    private static final Logger s_logger = new Logger(NuageVspGuruClientImpl.class);

    public NuageVspGuruClientImpl(NuageVspApiClient nuageVspApiClient, NuageVspAclClient nuageVspAclClient) {
        super(nuageVspApiClient, nuageVspAclClient);
    }

    @Override
    public void implement(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) throws NuageVspException {
        if (vspNetwork.isL2()) {
            s_logger.debug("Handling implement() call back API. Check with VSP to see if a  Isolated L2 networks exist or not");
            nuageVspApiClient.createIsolatedL2NetworkWithDefaultACLs(vspNetwork);
        } else {
            s_logger.debug("Handling implement() call back API. Check with VSP to see if a " + vspNetwork.getNetworkType().name() + " exist or not");
            nuageVspApiClient.createNetworkConfigurationWithDefaultACLs(vspNetwork, vspDhcpOptions);
        }
    }

    public void reserve(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, VspStaticNat vspStaticNat, VspDhcpVMOption vspDhcpVMOption) throws NuageVspException {
        NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        s_logger.debug("VSP Network to create VM or attach an interface is " + attachedNetworkDetails.getSubnetType() + "/" + attachedNetworkDetails.getSubnetId());

        NuageVspObject vm = nuageVspApiClient.getVMDetails(vspVm.getUuid());
        List<NuageVspObject> vmInterfaces;
        Pair<String, String> domainIdAndVPortId = null;
        //now execute all the APIs a the network's account user. So reset the nuage API parameters
        DhcpOptions dhcpOptions;
        if (vspDhcpVMOption != null && !vspDhcpVMOption.isDomainRouter()) {
            dhcpOptions = new DhcpOptions(vspDhcpVMOption.defaultHasDns(), vspDhcpVMOption.getHostname(),
                    vspDhcpVMOption.networkHasDns(), vspDhcpVMOption.isDefaultInterface(), vspDhcpVMOption.isDomainRouter());
        } else {
            dhcpOptions  = new DhcpOptions();
        }

        if (vm == null) {
            //VM does not exists in VSP. So, create the VM in VSP
            vmInterfaces = nuageVspApiClient.createVMInVSP(vspNetwork, vspVm, vspNic, attachedNetworkDetails, dhcpOptions);
            if (CollectionUtils.isNotEmpty(vmInterfaces)) {
                domainIdAndVPortId = getDomainIdAndVPort(vspNic.getMacAddress(), vmInterfaces);
            } else {
                throw new NuageVspApiException("Failed to get IP for the VM from VSP address for network " + vspNetwork.getUuid());
            }
        } else {
            //VM already exists, so just add the VM interface to the VM
            vmInterfaces = nuageVspApiClient.addVMInterfaceToVM(vspNetwork, vspVm, vspNic, attachedNetworkDetails, vm, dhcpOptions);
            if (CollectionUtils.isNotEmpty(vmInterfaces)) {
                domainIdAndVPortId = getDomainIdAndVPort(vspNic.getMacAddress(), vmInterfaces);
            } else {
                s_logger.info("Interface with MAC already exists in Nuage VSP. So, it is not added to the VM");
            }
        }

        if (vspStaticNat != null && vspVm.getDomainRouter() != Boolean.TRUE && vspStaticNat.getState() == VspStaticNat.State.Allocated && vspStaticNat.getOneToOneNat() == Boolean.TRUE) {
            s_logger.debug("Found a StaticNat(in ACS DB) " + vspStaticNat.getIpAddress() + " in Allocated state and it is associated to VM " + vspVm.getName()
                    + ". Trying to check if this StaticNAT is in sync with VSP.");
            if (vm != null && CollectionUtils.isEmpty(vmInterfaces)) {
                //Case were this is a VM restart
                vmInterfaces = vm.get(NuageVspAttribute.VM_INTERFACES);
                domainIdAndVPortId = getDomainIdAndVPort(vspNic.getMacAddress(), vmInterfaces);
            }
            try {
                if (domainIdAndVPortId != null) {
                    //Check if the VM in VSD has this StaticNAT and apply it if needed
                    nuageVspApiClient.applyStaticNatInVsp(attachedNetworkDetails, domainIdAndVPortId.getRight(), vspStaticNat);
                }
            } catch (NuageVspException e) {
                s_logger.warn("Post processing of StaticNAT could not continue. Error happened while checking if StaticNat " + vspStaticNat.getIpAddress()
                        + " is in Sync with VSP. " + e.getMessage());
            }
        }
    }

    private Pair<String, String> getDomainIdAndVPort(String nicMacAddress, List<NuageVspObject> vmInterfaces) throws NuageVspException {
        for (NuageVspObject vmInterface : vmInterfaces) {
            String macFromNuage = vmInterface.get(NuageVspAttribute.VM_INTERFACE_MAC);
            if (StringUtils.equals(macFromNuage, nicMacAddress)) {
                String domainId = vmInterface.get(NuageVspAttribute.VM_INTERFACE_DOMAIN_ID);
                String vportId = vmInterface.get(NuageVspAttribute.VM_INTERFACE_VPORT_ID);
                return Pair.of(domainId, vportId);
            }
        }
        return null;
    }

    public void deallocate(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic) throws NuageVspException {
        NuageVspObject vm = nuageVspApiClient.getVMDetails(vspVm.getUuid());

        if (vm != null) {
            //get the VM ID
            List<NuageVspObject> vmInterfaceDetails = vm.get(NuageVspAttribute.VM_INTERFACES);
            String vmId = vm.get(NuageVspAttribute.ID);

            if (vmInterfaceDetails != null && vmInterfaceDetails.size() > 0) {
                // This is a case of a NIC being removed.
                NuageVspObject deletedVmInterfaceDetail = null;
                String macAddr = null;
                for (NuageVspObject vmInterfaceDetail : vmInterfaceDetails) {
                    macAddr = vmInterfaceDetail.get(NuageVspAttribute.VM_INTERFACE_MAC);
                    if (macAddr.equals(vspNic.getMacAddress())) {
                        deletedVmInterfaceDetail = vmInterfaceDetail;
                        break;
                    }
                }
                if (deletedVmInterfaceDetail != null) {
                    String vmInterfaceID = deletedVmInterfaceDetail.get(NuageVspAttribute.ID);
                    String vPortId = deletedVmInterfaceDetail.get(NuageVspAttribute.VM_INTERFACE_VPORT_ID);
                    nuageVspApiClient.deleteVmInterface(vspVm.getUuid(), macAddr, vmInterfaceID);
                    s_logger.debug("Handling deallocate(). Deleted VM interface with IP " + vspNic.getIp() + " for VM " + vspVm.getName() + " with MAC " + macAddr
                            + " from VM with UUID " + vspVm.getUuid());

                    if (vspNetwork.isShared() && vspNetwork.isPublicAccess()) {
                        NetworkDetails networkDetails = getAttachedNetworkDetails(vspNetwork);
                        nuageVspApiClient.releaseFIPFromVsp(networkDetails, vPortId, (String) deletedVmInterfaceDetail.getExternalId(), null);
                    }

                    //clean up the stale Vport object that is attached to the VMInterface
                    nuageVspApiClient.deleteQuietly(NuageVspEntity.VPORT, vPortId);
                } else {
                    s_logger.debug("Handling deallocate(). VM," + vspVm.getName() + ", interface with IP " + vspNic.getIp() + " and MAC " + vspNic.getMacAddress() + " is not present in VSP");
                }
            }

            if ((vmInterfaceDetails == null || vmInterfaceDetails.size() == 0 || vmInterfaceDetails.size() == 1)
                    && vspVm.getState() == VspVm.State.Expunging) {
                //cleanup the VM as there are no VMInterfaces...
                s_logger.debug("Handling deallocate(). VM," + vspVm.getName() + ", with interface IP "
                        + vspNic.getIp() + " and MAC " + vspNic.getMacAddress() + " is in Expunging state. So, delete this VM from VSP");
                nuageVspApiClient.deleteVM(vspVm.getUuid(), vmId);
            }
        } else {
            cleanVMVPorts(vspNetwork, vspVm, vspNic);
        }
    }

    private void cleanVMVPorts(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic) throws NuageVspException {
        s_logger.debug("Handling deallocate(). VM " + vspVm.getName() + "  with NIC IP " + vspNic.getIp() + " is getting destroyed and it does not exists in NuageVSP.");
        //Clean up the VPorts
        NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(false, vspNetwork);
        if (attachedNetworkDetails == null) {
            s_logger.debug("The network details corresponding to network " + vspNetwork.getUuid() + " could not be found.");
            return;
        }

        String vportId = nuageVspApiClient.findEntityIdByExternalUuid(attachedNetworkDetails.getSubnetType(), attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT, vspNic.getUuid());
        if (StringUtils.isNotBlank(vportId)) {
            nuageVspApiClient.deleteQuietly(NuageVspEntity.VPORT, vportId);
        }
    }

    public void trash(VspNetwork vspNetwork) throws NuageVspException {
        // Shared networks can exist under multiple enterprises, delete from all
        if (vspNetwork.isShared()) {
            String enterprisesJson = nuageVspApiClient.getResources(NuageVspEntity.ENTERPRISE);
            List<NuageVspObject> enterprises = nuageVspApiClient.parseJsonString(NuageVspEntity.ENTERPRISE, enterprisesJson);
            for (NuageVspObject enterprise : enterprises) {
                String enterpriseId = enterprise.get(NuageVspAttribute.ID);
                nuageVspApiClient.cleanUpDomainAndTemplate(enterpriseId, vspNetwork.getUuid(), vspNetwork.getDomainTemplateName());
            }

            for (VspAddressRange addressRange : vspNetwork.getAddressRanges()) {
                String externalIdFromGateway = UuidUtils.generateUuidFromExternalIdAndIp(vspNetwork.getUuid(), addressRange.getGateway());
                nuageVspApiClient.deleteSharedResourceInVSP(SharedNetworkType.FLOATING, externalIdFromGateway, addressRange.getGateway(), addressRange.getNetmask());
                nuageVspApiClient.deleteSharedResourceInVSP(SharedNetworkType.PUBLIC, externalIdFromGateway, addressRange.getGateway(), addressRange.getNetmask());
            }

            return;
        }

        String enterpriseId = nuageVspApiClient.getEnterprise(vspNetwork.getVspDomain().getUuid());
        if (StringUtils.isNotBlank(enterpriseId)) {
            if (vspNetwork.isL3() || vspNetwork.isVpc()) {
                if (vspNetwork.isVpc()) {
                    String vspDomainId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN, vspNetwork.getVpcUuid());
                    if (StringUtils.isNotBlank(vspDomainId)) {
                        String vspNetworkId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.DOMAIN, vspDomainId, NuageVspEntity.SUBNET, vspNetwork.getSubnetExternalId());
                        if (StringUtils.isNotBlank(vspNetworkId)) {
                            s_logger.debug("Found a VSP L3 network " + vspNetworkId + " that corresponds to tier " + vspNetwork.getUuid() + " in CS. So, delete it");
                            nuageVspApiClient.deleteQuietly(NuageVspEntity.SUBNET, vspNetworkId);
                        }
                    }
                } else {
                    nuageVspApiClient.cleanUpDomainAndTemplate(enterpriseId, vspNetwork.getUuid(), vspNetwork.getDomainTemplateName());
                }
            } else {
                String vspNetworkId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.L2DOMAIN_TEMPLATE, vspNetwork.getUuid());
                if (StringUtils.isNotBlank(vspNetworkId)) {
                    nuageVspApiClient.deleteQuietly(NuageVspEntity.L2DOMAIN_TEMPLATE, vspNetworkId);
                    s_logger.debug("Found a VSP L2 network " + vspNetworkId + " that corresponds to network " + vspNetwork.getUuid() + " in CS and deleted it");
                }
            }
        }
    }

    @Override
    public void applyDhcpOptions(List<VspDhcpVMOption> vspDhcpOptions, VspNetwork vspNetwork) throws NuageVspException {
        NetworkDetails attachedNetworkDetails = getAttachedNetworkDetails(vspNetwork);
        for (VspDhcpVMOption vspDhcpOption : vspDhcpOptions) {
            DhcpOptions dhcpOptions = new DhcpOptions(vspDhcpOption.defaultHasDns(), vspDhcpOption.getHostname(), vspDhcpOption.networkHasDns(), vspDhcpOption.isDefaultInterface(), vspDhcpOption.isDomainRouter());
            String vportVsdId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.SUBNET, attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT, vspDhcpOption.getNicUuid());
            nuageVspApiClient.createDhcpOptions(NuageVspApiClient.ExistingDhcpOptionStrategy.FETCH_AND_DELETE,
                    NuageVspEntity.VPORT, vspDhcpOption.getNicUuid(), vportVsdId, vspNetwork, dhcpOptions);
        }
    }
}
