/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.model;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 *
 */
public class Protocol {
    public final static Protocol TCP = new Protocol("6");
    public final static Protocol UDP = new Protocol("17");
    public final static Protocol ICMP = new Protocol("1");
    public final static Protocol ANY = new Protocol("ANY");

    private final static Map<String, Protocol> PROTOCOL_MAP = new HashMap<String, Protocol>();

    static {
        PROTOCOL_MAP.put("TCP", TCP);
        PROTOCOL_MAP.put("UDP", UDP);
        PROTOCOL_MAP.put("ICMP", ICMP);
        PROTOCOL_MAP.put("ANY", ANY);
        PROTOCOL_MAP.put("ALL", ANY);

        PROTOCOL_MAP.put("6", TCP);
        PROTOCOL_MAP.put("17", UDP);
        PROTOCOL_MAP.put("1", ICMP);
    }

    @Override
    public boolean equals(Object other) {
        return EqualsBuilder.reflectionEquals(this, other);
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    private String protocolNumber;

    private Protocol(String protocolNumber) {
        this.protocolNumber = protocolNumber;
    }

    public String getProtocolNumber() {
        return protocolNumber;
    }

    public boolean hasPort() {
        return this.equals(TCP) || this.equals(UDP);
    }

    public static Protocol fromString(String protocolString) {
        Protocol protocol = PROTOCOL_MAP.get(protocolString.toUpperCase());

        if (protocol == null) {
            protocol = new Protocol(protocolString);
        }

        return protocol;
    }

    @Override public String toString() {
        return new ToStringBuilder(this)
                .append("protocolNumber", protocolNumber)
                .toString();
    }
}
