/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.model;

import java.util.List;

import net.nuage.vsp.acs.client.common.NuageVspConstants;

import com.google.common.collect.Lists;

public class VspAclRule extends VspObject {

    public enum ACLState {
        Active, Add, Revoke
    }

    public enum ACLTrafficType {
        Ingress, Egress
    }

    public enum ACLAction {
        Allow, Deny
    }

    public enum ACLType {
        Firewall, NetworkACL
    }

    private final String uuid;
    private final Protocol protocol;
    private final Integer startPort;
    private final Integer endPort;
    private final ACLState state;
    private final ACLTrafficType trafficType;
    private final ACLAction action;
    private final String sourceIpAddress;
    private final List<String> sourceCidrList;
    private final Integer priority;
    private final ACLType type;

    private VspAclRule(String uuid, Protocol protocol, Integer startPort, Integer endPort, ACLState state, ACLTrafficType trafficType,
                       ACLAction action, String sourceIpAddress, List<String> sourceCidrList, Integer priority, ACLType type) {
        this.uuid = uuid;
        this.protocol = protocol;
        this.startPort = startPort;
        this.endPort = endPort;
        this.state = state;
        this.trafficType = trafficType;
        this.action = action;
        this.sourceIpAddress = sourceIpAddress;
        this.sourceCidrList = sourceCidrList;
        this.priority = priority;
        this.type = type;
    }

    public String getUuid() {
        return uuid;
    }

    public Protocol getProtocol() {
        return protocol;
    }

    public Integer getStartPort() {
        return startPort;
    }

    public Integer getEndPort() {
        return endPort;
    }

    public ACLState getState() {
        return state;
    }

    public ACLTrafficType getTrafficType() {
        return trafficType;
    }

    public ACLAction getAction() {
        return action;
    }

    public String getSourceIpAddress() {
        return sourceIpAddress;
    }

    public List<String> getSourceCidrList() {
        return sourceCidrList != null ? Lists.newCopyOnWriteArrayList(sourceCidrList) : null;
    }

    public Integer getPriority() {
        return priority;
    }

    public ACLType getType() {
        return type;
    }

    public boolean isReflexive() {
        return action == ACLAction.Allow && protocol.hasPort();
    }

    public boolean isAddedNetworkAclRule() {
        return state == ACLState.Add && type == ACLType.NetworkACL;
    }

    public String getPortRange() {
        if (!protocol.hasPort()) {
            return null;
        }

        String portRange = NuageVspConstants.STAR;

        if (startPort != null && endPort != null ) {
            portRange = String.format("%d-%d", startPort, endPort);
        } else if (startPort != null) {
            portRange = String.valueOf(startPort);
        }

        return portRange;
    }

    public static class Builder extends VspBuilder<Builder, VspAclRule> {
        private String uuid;
        private Protocol protocol;
        private Integer startPort;
        private Integer endPort;
        private ACLState state;
        private ACLTrafficType trafficType;
        private ACLAction action;
        private String sourceIpAddress;
        private List<String> sourceCidrList;
        private Integer priority;
        private ACLType type;

        public Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public Builder protocol(Protocol protocol) {
            this.protocol = protocol;
            return this;
        }

        public Builder protocol(String protocol) {
            this.protocol = Protocol.fromString(protocol);
            return this;
        }

        public Builder startPort(Integer startPort) {
            this.startPort = startPort;
            return this;
        }

        public Builder endPort(Integer endPort) {
            this.endPort = endPort;
            return this;
        }

        public Builder state(ACLState state) {
            this.state = state;
            return this;
        }

        public Builder trafficType(ACLTrafficType trafficType) {
            this.trafficType = trafficType;
            return this;
        }

        public Builder action(ACLAction action) {
            this.action = action;
            return this;
        }

        public Builder sourceIpAddress(String sourceIpAddress) {
            this.sourceIpAddress = sourceIpAddress;
            return this;
        }

        public Builder sourceCidrList(List<String> sourceCidrList) {
            this.sourceCidrList = sourceCidrList;
            return this;
        }

        public Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public Builder type(ACLType type) {
            this.type = type;
            return this;
        }

        @Override
        public VspAclRule build() {
            return new VspAclRule(uuid, protocol, startPort, endPort, state, trafficType, action,
                    sourceIpAddress, sourceCidrList, priority, type);
        }

        @Override
        public Builder fromObject(VspAclRule obj) {
            return new Builder().uuid(obj.getUuid()).protocol(obj.getProtocol()).startPort(obj.getStartPort()).endPort(obj.getEndPort())
                    .state(obj.getState()).trafficType(obj.getTrafficType()).action(obj.getAction()).sourceIpAddress(obj.getSourceIpAddress())
                    .sourceCidrList(obj.getSourceCidrList()).priority(obj.getPriority()).type(obj.getType());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof VspAclRule)) return false;

        VspAclRule that = (VspAclRule) o;

        if (action != that.action) return false;
        if (endPort != null ? !endPort.equals(that.endPort) : that.endPort != null) return false;
        if (priority != null ? !priority.equals(that.priority) : that.priority != null) return false;
        if (protocol != null ? !protocol.equals(that.protocol) : that.protocol != null) return false;
        if (sourceCidrList != null ? !sourceCidrList.equals(that.sourceCidrList) : that.sourceCidrList != null)
            return false;
        if (sourceIpAddress != null ? !sourceIpAddress.equals(that.sourceIpAddress) : that.sourceIpAddress != null)
            return false;
        if (startPort != null ? !startPort.equals(that.startPort) : that.startPort != null) return false;
        if (state != that.state) return false;
        if (trafficType != that.trafficType) return false;
        if (type != that.type) return false;
        if (uuid != null ? !uuid.equals(that.uuid) : that.uuid != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = uuid != null ? uuid.hashCode() : 0;
        result = 31 * result + (protocol != null ? protocol.hashCode() : 0);
        result = 31 * result + (startPort != null ? startPort.hashCode() : 0);
        result = 31 * result + (endPort != null ? endPort.hashCode() : 0);
        result = 31 * result + (state != null ? state.hashCode() : 0);
        result = 31 * result + (trafficType != null ? trafficType.hashCode() : 0);
        result = 31 * result + (action != null ? action.hashCode() : 0);
        result = 31 * result + (sourceIpAddress != null ? sourceIpAddress.hashCode() : 0);
        result = 31 * result + (sourceCidrList != null ? sourceCidrList.hashCode() : 0);
        result = 31 * result + (priority != null ? priority.hashCode() : 0);
        result = 31 * result + (type != null ? type.hashCode() : 0);
        return result;
    }
}