/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.model;

import java.util.List;

import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.common.utils.UuidUtils;

public class VspNetwork extends VspObject {
    public enum NetworkType {
        L2, L3, Shared, Vpc
    }

    private final long id;
    private final String uuid;
    private final String name;
    private final VspDomain vspDomain;
    private final String accountName;
    private final String accountUuid;
    private final String vpcUuid;
    private final String vpcName;
    private final NetworkType networkType;
    private final boolean firewallServiceSupported;
    private final boolean egressDefaultPolicy;
    private final String domainTemplateName;
    private final String cidr;
    private final String gateway;
    private final String virtualRouterIp;
    private final boolean publicAccess;
    private final List<VspAddressRange> addressRanges;
    private final Pair<String, String> vpcOrSubnetInfo;
    private final boolean vlanUnderlay;

    private VspNetwork(long id, String uuid, String name, VspDomain vspDomain, String accountName, String accountUuid, String vpcUuid,
                       String vpcName, NetworkType networkType, boolean firewallServiceSupported, boolean egressDefaultPolicy,
                       String domainTemplateName, String cidr, String gateway, String virtualRouterIp, boolean publicAccess,
                       List<VspAddressRange> addressRanges, boolean vlanUnderlay) {
        this.id = id;
        this.uuid = uuid;
        this.name = name;
        this.vspDomain = vspDomain;
        this.accountName = accountName;
        this.accountUuid = accountUuid;
        this.vpcUuid = vpcUuid;
        this.vpcName = vpcName;
        this.networkType = networkType;
        this.firewallServiceSupported = firewallServiceSupported;
        this.egressDefaultPolicy = egressDefaultPolicy;
        this.domainTemplateName = domainTemplateName;
        this.cidr = cidr;
        this.gateway = gateway;
        this.virtualRouterIp = virtualRouterIp;
        this.publicAccess = publicAccess;
        this.addressRanges = addressRanges;
        this.vlanUnderlay = vlanUnderlay;

        if (isVpc()) {
            vpcOrSubnetInfo = Pair.of(this.vpcUuid, "VPC_" + this.vpcName);
        } else {
            vpcOrSubnetInfo = Pair.of(this.uuid, this.name);
        }
    }

    public long getId() {
        return id;
    }

    public String getUuid() {
        return uuid;
    }

    public String getName() {
        return name;
    }

    public VspDomain getVspDomain() {
        return vspDomain;
    }

    public String getAccountName() {
        return accountName;
    }

    public String getAccountUuid() {
        return accountUuid;
    }

    public String getVpcUuid() {
        return vpcUuid;
    }

    public String getVpcName() {
        return vpcName;
    }

    public boolean isL2() {
        return this.networkType == NetworkType.L2;
    }

    public boolean isL3() {
        return this.networkType == NetworkType.L3;
    }

    public boolean isShared() {
        return this.networkType == NetworkType.Shared;
    }

    public boolean isVpc() {
        return this.networkType == NetworkType.Vpc;
    }

    public NetworkType getNetworkType() {
        return networkType;
    }

    public boolean isFirewallServiceSupported() {
        return firewallServiceSupported;
    }

    public boolean isEgressDefaultPolicy() {
        return egressDefaultPolicy;
    }

    public String getDomainTemplateName() {
        return domainTemplateName;
    }

    public String getCidr() {
        return cidr;
    }

    public String getGateway() {
        return gateway;
    }

    public String getVirtualRouterIp() {
        return virtualRouterIp;
    }

    public boolean isPublicAccess() {
        return publicAccess;
    }

    public List<VspAddressRange> getAddressRanges() {
        return addressRanges;
    }

    public Pair<String, String> getVpcOrSubnetInfo() {
        return vpcOrSubnetInfo;
    }

    public boolean shouldReuseDomain() {
        return networkType == NetworkType.Vpc || networkType == NetworkType.Shared;
    }

    public String getSubnetExternalId() {
        return isShared() ? UuidUtils.generateUuidFromExternalIdAndIp(uuid, gateway) : uuid;
    }
    public boolean isVlanUnderlay() {
        return vlanUnderlay;
    }

    public NuageVspUser getProxyUser() {
        return new NuageVspUser(getAccountUuid(), getVspDomain().getUuid());
    }

    public static class Builder extends VspBuilder<Builder, VspNetwork> {
        private long id;
        private String uuid;
        private String name;
        private VspDomain vspDomain;
        private String accountName;
        private String accountUuid;
        private String vpcUuid;
        private String vpcName;
        private NetworkType networkType;
        private boolean firewallServiceSupported;
        private boolean egressDefaultPolicy;
        private String domainTemplateName;
        private String cidr;
        private String gateway;
        private String virtualRouterIp;
        private boolean publicAccess;
        private List<VspAddressRange> addressRanges;
        private boolean vlanUnderlay;

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder domain(VspDomain vspDomain) {
            this.vspDomain = vspDomain;
            return this;
        }

        public Builder accountName(String accountName) {
            this.accountName = accountName;
            return this;
        }

        public Builder accountUuid(String accountUuid) {
            this.accountUuid = accountUuid;
            return this;
        }

        public Builder vpcUuid(String vpcUuid) {
            this.vpcUuid = vpcUuid;
            return this;
        }

        public Builder vpcName(String vpcName) {
            this.vpcName = vpcName;
            return this;
        }

        public Builder networkType(NetworkType networkType) {
            this.networkType = networkType;
            return this;
        }

        public Builder firewallServiceSupported(boolean firewallServiceSupported) {
            this.firewallServiceSupported = firewallServiceSupported;
            return this;
        }

        public Builder egressDefaultPolicy(boolean egressDefaultPolicy) {
            this.egressDefaultPolicy = egressDefaultPolicy;
            return this;
        }

        public Builder domainTemplateName(String domainTemplateName) {
            this.domainTemplateName = domainTemplateName;
            return this;
        }

        public Builder cidr(String cidr) {
            this.cidr = cidr;
            return this;
        }

        public Builder gateway(String gateway) {
            this.gateway = gateway;
            return this;
        }

        public Builder virtualRouterIp(String virtualRouterIp) {
            this.virtualRouterIp = virtualRouterIp;
            return this;
        }

        public Builder publicAccess(boolean publicAccess) {
            this.publicAccess = publicAccess;
            return this;
        }

        public Builder addressRanges(List<VspAddressRange> addressRanges) {
            this.addressRanges = addressRanges;
            return this;
        }

        public Builder vlanUnderlay(boolean vlanUnderlay) {
            this.vlanUnderlay = vlanUnderlay;
            return this;
        }

        @Override
        public VspNetwork build() {
            return new VspNetwork(id, uuid, name, vspDomain, accountName, accountUuid, vpcUuid, vpcName, networkType,
                    firewallServiceSupported, egressDefaultPolicy, domainTemplateName, cidr, gateway, virtualRouterIp,
                    publicAccess, addressRanges, vlanUnderlay);
        }

        @Override
        public Builder fromObject(VspNetwork obj) {
            return new Builder().id(obj.getId()).uuid(obj.getUuid()).name(obj.getName()).domain(obj.getVspDomain())
                    .accountName(obj.getAccountName()).accountUuid(obj.getAccountUuid()).vpcUuid(obj.getVpcUuid())
                    .vpcName(obj.getVpcName()).networkType(obj.getNetworkType()).firewallServiceSupported(obj.isFirewallServiceSupported())
                    .egressDefaultPolicy(obj.isEgressDefaultPolicy()).domainTemplateName(obj.getDomainTemplateName())
                    .cidr(obj.getCidr()).gateway(obj.getGateway()).virtualRouterIp(obj.getVirtualRouterIp())
                    .publicAccess(obj.isPublicAccess()).addressRanges(obj.getAddressRanges()).vlanUnderlay(obj.isVlanUnderlay());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof VspNetwork)) return false;

        VspNetwork that = (VspNetwork) o;

        if (egressDefaultPolicy != that.egressDefaultPolicy) return false;
        if (firewallServiceSupported != that.firewallServiceSupported) return false;
        if (id != that.id) return false;
        if (publicAccess != that.publicAccess) return false;
        if (accountName != null ? !accountName.equals(that.accountName) : that.accountName != null) return false;
        if (accountUuid != null ? !accountUuid.equals(that.accountUuid) : that.accountUuid != null) return false;
        if (addressRanges != null ? !addressRanges.equals(that.addressRanges) : that.addressRanges != null)
            return false;
        if (cidr != null ? !cidr.equals(that.cidr) : that.cidr != null) return false;
        if (domainTemplateName != null ? !domainTemplateName.equals(that.domainTemplateName) : that.domainTemplateName != null)
            return false;
        if (gateway != null ? !gateway.equals(that.gateway) : that.gateway != null) return false;
        if (name != null ? !name.equals(that.name) : that.name != null) return false;
        if (networkType != that.networkType) return false;
        if (uuid != null ? !uuid.equals(that.uuid) : that.uuid != null) return false;
        if (virtualRouterIp != null ? !virtualRouterIp.equals(that.virtualRouterIp) : that.virtualRouterIp != null)
            return false;
        if (vpcName != null ? !vpcName.equals(that.vpcName) : that.vpcName != null) return false;
        if (vpcUuid != null ? !vpcUuid.equals(that.vpcUuid) : that.vpcUuid != null) return false;
        if (vspDomain != null ? !vspDomain.equals(that.vspDomain) : that.vspDomain != null) return false;
        if (vlanUnderlay != that.vlanUnderlay) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = (int) (id ^ (id >>> 32));
        result = 31 * result + (uuid != null ? uuid.hashCode() : 0);
        result = 31 * result + (name != null ? name.hashCode() : 0);
        result = 31 * result + (vspDomain != null ? vspDomain.hashCode() : 0);
        result = 31 * result + (accountName != null ? accountName.hashCode() : 0);
        result = 31 * result + (accountUuid != null ? accountUuid.hashCode() : 0);
        result = 31 * result + (vpcUuid != null ? vpcUuid.hashCode() : 0);
        result = 31 * result + (vpcName != null ? vpcName.hashCode() : 0);
        result = 31 * result + (networkType != null ? networkType.hashCode() : 0);
        result = 31 * result + (firewallServiceSupported ? 1 : 0);
        result = 31 * result + (egressDefaultPolicy ? 1 : 0);
        result = 31 * result + (domainTemplateName != null ? domainTemplateName.hashCode() : 0);
        result = 31 * result + (cidr != null ? cidr.hashCode() : 0);
        result = 31 * result + (gateway != null ? gateway.hashCode() : 0);
        result = 31 * result + (virtualRouterIp != null ? virtualRouterIp.hashCode() : 0);
        result = 31 * result + (publicAccess ? 1 : 0);
        result = 31 * result + (addressRanges != null ? addressRanges.hashCode() : 0);
        result = 31 * result + (vlanUnderlay ? 1 : 0);
        return result;
    }
}
