/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common;

import net.nuage.vsp.acs.client.common.utils.Logger;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

/**
 * Utility class to access the VSP configuration properties.
 */
public final class ConfigUtil {
    private static final Logger logger = new Logger(ConfigUtil.class);

    /**
     * VSP default configuration file path to override using -Dvsp.default.config
     * property.
     */
    private static final String VSP_CONFIG_FILENAME_PROPERTY = "vsp.default.config";

    /**
     * The config properties file name.
     */
    private static final String CONFIG_FILE = "/META-INF/vsp-defaults.properties";

    /**
     * Default config parameters
     */
    public static final String FLOATING_IP_QUOTA = "floatingIPQuota";
    public static final String ALLOW_GATEWAY_MANAGEMENT = "allowGatewayMgmt";
    public static final String ALLOW_ADVANCED_QOS = "allowAdvancedQOS";
    public static final String AVAILABLE_FWD_CLASS = "availableFwdClass";

    private static final Properties properties = new Properties();

    static {
        try {
            String path = System.getProperty(VSP_CONFIG_FILENAME_PROPERTY);
            InputStream in = null;
            if (path != null) {
                in = new BufferedInputStream(new FileInputStream(path));
            } else {
                path = CONFIG_FILE;
                URL url = ConfigUtil.class.getClassLoader().getResource(path);
                if (url != null) {
                    in = url.openStream();
                }
            }

            if (in != null) {
                properties.load(in);
                in.close();
            } else {
                logger.warn("Failed to load the config file " + path);
            }
        } catch (Throwable e) {
            logger.error("Failed to load the config file.", e);
        }
    }

    public static String getProperty(String key, String defaultValue) {
        return properties.getProperty(key, defaultValue);
    }

    public static int getPropertyInteger(String key, int defaultValue) {
        String value = properties.getProperty(key);
        if (value == null) {
            return defaultValue;
        }
        int intValue;
        try {
            intValue = Integer.parseInt(value);
        } catch (NumberFormatException e) {
            logger.error("invalid value format for key = " + key + " value = " + value, e);
            intValue = defaultValue;
        }
        return intValue;
    }

    public static boolean getPropertyBoolean(String key, boolean defaultValue) {
        String value = properties.getProperty(key);
        if (value == null) {
            return defaultValue;
        }
        boolean booleanValue;
        try {
            booleanValue = Boolean.parseBoolean(value);
        } catch (Exception e) {
            logger.error("invalid value format for key = " + key + " value = " + value, e);
            booleanValue = defaultValue;
        }
        return booleanValue;
    }
}
