/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common;

import java.util.Comparator;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jdk.nashorn.internal.runtime.options.Option;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang3.StringUtils;

public class NuageVspApiVersion implements Comparable<NuageVspApiVersion> {
    public enum Status { UNSUPPORTED, DEPRECATED, CURRENT };

    private static final Pattern versionRegEx = Pattern.compile("v(\\d+)_(\\d+)(?:_(\\d+))?");
    public static final NuageVspApiVersion V3_2 = new NuageVspApiVersion(3, 2);
    public static final NuageVspApiVersion V4_0 = new NuageVspApiVersion(4, 0);

    public static final NuageVspApiVersion CURRENT = V4_0;

    public static final NuageVspApiVersion[] SUPPORTED_VERSIONS = new NuageVspApiVersion[] {V3_2, V4_0};
    public static final NuageVspApiVersion[] VERSIONS_REQUIRING_UPGRADE = new NuageVspApiVersion[] {V3_2, V4_0};

    private Integer major;
    private Integer minor;
    private final Optional<Integer> patch;

    public NuageVspApiVersion(int major, int minor) {
        this.major = major;
        this.minor = minor;
        this.patch = Optional.empty();
    }

    public NuageVspApiVersion(int major, int minor, int patch) {
        this.major = major;
        this.minor = minor;
        this.patch = Optional.of(patch);
    }

    public NuageVspApiVersion(String version) {
        final Matcher matcher = versionRegEx.matcher(version);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Bad version");
        }

        major = Integer.valueOf(matcher.group(1));
        minor = Integer.valueOf(matcher.group(2));
        patch = Optional.ofNullable(matcher.group(3))
                        .map(Integer::valueOf);
    }

    public static NuageVspApiVersion fromString(String version){
        version = version.toLowerCase().replace('.','_');

        for (NuageVspApiVersion suppVersion: SUPPORTED_VERSIONS) {
            if (StringUtils.equals(suppVersion.toString(), version)) {
                return suppVersion;
            }
        }
        return new NuageVspApiVersion(version);
    }

    @Override
    public boolean equals(Object other) {
        return EqualsBuilder.reflectionEquals(this, other);
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public int compareTo(NuageVspApiVersion other) {
        return Comparator.comparingInt((NuageVspApiVersion v) -> v.major)
                .thenComparing((NuageVspApiVersion v) -> v.minor)
                .thenComparing((NuageVspApiVersion v) -> v.patch.orElse(0))
                .compare(this, other);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("v")
          .append(major)
          .append("_")
          .append(minor);

        patch.ifPresent(p -> sb.append("_").append(p));

        return sb.toString();
    }

    public boolean isSupported() {
        return ArrayUtils.contains(SUPPORTED_VERSIONS, this);
    }
}
