/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.nuage.vsp.acs.client.api.model.VspAclRule;

import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;

public class AclRulesDetails {

    private final Set<String> ruleUuids;

    enum AclType {
        ACL,
        FW_INGRESS,
        FW_EGRESS
    }

    private final List<VspAclRule> vspAclRules;
    private final boolean isNetworkAcl;
    private final Boolean isIngressFirewallRule;
    private final boolean createDefaultRules;
    private final String aclNetworkLocationId;
    private final NuageVspObject ingressAclTemplate;
    private final NuageVspObject egressAclTemplate;
    private final boolean allRulesActive;
    private final boolean needsBlockRule;

    private final static Predicate<VspAclRule> IS_ACTIVE_RULE = new Predicate<VspAclRule>() {
        @Override public boolean apply(VspAclRule input) {
            return input.getState() == VspAclRule.ACLState.Active;
        }
    };

    private final static Predicate<VspAclRule> IS_EGRESS_ALLOW = new Predicate<VspAclRule>() {
        @Override public boolean apply(VspAclRule input) {
            return input.getState() != VspAclRule.ACLState.Revoke
                    && input.getAction() == VspAclRule.ACLAction.Allow
                    && input.getTrafficType() == VspAclRule.ACLTrafficType.Egress;
        }
    };

    private Set<String> getRuleUuids() {
        Set<String> result = new HashSet<String>();

        if(vspAclRules != null) {
            for (VspAclRule rule : vspAclRules) {
                result.add(String.format("%s_%s", rule.getTrafficType(), rule.getUuid()));
            }
        }

        return result;

    }

    public AclRulesDetails(List<VspAclRule> vspAclRules, boolean isNetworkAcl, Boolean isIngressFirewallRule, boolean createDefaultRules, String aclNetworkLocationId,
            NuageVspObject ingressAclTemplate, NuageVspObject egressAclTemplate) {
        this.vspAclRules = vspAclRules;
        this.isNetworkAcl = isNetworkAcl;
        this.isIngressFirewallRule = isIngressFirewallRule;
        this.createDefaultRules = createDefaultRules;
        this.aclNetworkLocationId = aclNetworkLocationId;
        this.ingressAclTemplate = ingressAclTemplate;
        this.egressAclTemplate = egressAclTemplate;
        this.allRulesActive = FluentIterable.from(vspAclRules).allMatch(IS_ACTIVE_RULE);
        this.needsBlockRule = isNetworkAcl && FluentIterable.from(vspAclRules).anyMatch(IS_EGRESS_ALLOW);
        this.ruleUuids = getRuleUuids();
    }

    public List<VspAclRule> getVspAclRules() {
        return vspAclRules;
    }

    public boolean isNetworkAcl() {
        return isNetworkAcl;
    }

    public Boolean isIngressFirewallRule() {
        return isIngressFirewallRule;
    }

    public boolean isCreateDefaultRules() {
        return createDefaultRules;
    }

    public String getAclNetworkLocationId() {
        return aclNetworkLocationId;
    }

    public NuageVspObject getIngressAclTemplate() {
        return ingressAclTemplate;
    }

    public NuageVspObject getEgressAclTemplate() {
        return egressAclTemplate;
    }

    public String getIngressAclTemplateId() {
        return ingressAclTemplate.getId();
    }

    public String getEgressAclTemplateId() {
        return egressAclTemplate.getId();
    }

    public boolean isNeedsBlockRule() {
        return needsBlockRule;
    }

    public boolean isAllRulesActive() {
        return allRulesActive;
    }

    public boolean isMissingIngressRule(NuageVspObject rule) {
        return !ruleUuids.contains("Ingress_" + rule.getExternalId());
    }

    public boolean isMissingEgressRule(NuageVspObject rule) {
        return !ruleUuids.contains("Egress_" + rule.getExternalId());
    }

    public boolean hasIngressRules() {
        return isIngressFirewallRule == null || isIngressFirewallRule == Boolean.TRUE;
    }

    public boolean hasEgressRules() {
        return isIngressFirewallRule == null || isIngressFirewallRule == Boolean.FALSE;
    }
}
