/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.util.Arrays;
import java.util.Formatter;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.lang.ArrayUtils;

import com.google.common.net.InetAddresses;
import com.google.common.primitives.Bytes;

import javax.xml.bind.DatatypeConverter;

public class DhcpOption {
    private String hexCode;
    private String hexLength;
    private String hexValue;

    private int code;
    private List<String> value;
    private Dhcp.DhcpOptionType optionType;


    public DhcpOption(int code, byte b) {
        setCode(code);
        setHexValue(new byte[] {b});
    }

    public DhcpOption(int code, String value) {
        setCode(code);
        this.value = new LinkedList<>();
        this.optionType = Dhcp.dhcpCodeToType.get(Dhcp.DhcpOptionCode.valueOf(code));

        if(optionType == Dhcp.DhcpOptionType.LIST_OF_IPv4 || optionType == Dhcp.DhcpOptionType.LIST_OF_STRINGS) {
            this.value.addAll(Arrays.asList(value.split(",")));
        } else {
            this.value.add(value);
        }

        if(optionType == Dhcp.DhcpOptionType.RAW) {
            //setHexValue(value.getBytes(Charset.forName("UTF-8")));
            setHexValue(DatatypeConverter.parseHexBinary(value));
        }
    }

    public DhcpOption(int code, List<String> ipAdresses) {
        setCode(code);
        this.value = ipAdresses; //Joiner.on(",").join(ipAdresses);
        this.optionType = Dhcp.dhcpCodeToType.get(Dhcp.DhcpOptionCode.valueOf(code));

        if(optionType == Dhcp.DhcpOptionType.RAW) {
            LinkedList<Byte> bytes = new LinkedList<Byte>();

            for (String ip : ipAdresses) {
                bytes.addAll(Arrays.asList(ArrayUtils.toObject(InetAddresses.forString(ip).getAddress())));
            }
            setHexValue(Bytes.toArray(bytes));
        }
    }

    public String getCodeAsHex() {
        return hexCode;
    }

    public int getCode() {
        return code;
    }

    public String getLengthAsHex() {
        return hexLength;
    }

    public String getValueAsHex() {
        return hexValue;
    }

    public List<String> getValue() {
        return value;
    }

    public Dhcp.DhcpOptionType getOptionType() {
        return optionType;
    }

    public void setOptionType(Dhcp.DhcpOptionType optionType) {
        this.optionType = optionType;
    }

    private void setHexValue(byte[] value) {
        this.hexLength = String.format("%02x", value.length);
        Formatter formatter = new Formatter();
        for (byte b : value) {
            formatter.format("%02x", b);
        }
        this.hexValue = formatter.toString();
    }

    private void setCode(int code) {
        this.code = code;
        this.hexCode = String.format("%02x", code);
    }


}
