/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.nuage.vsp.acs.client.api.model.Protocol;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.gson.NuageVspObjectTypeAdapter;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.gson.reflect.TypeToken;

public enum NuageVspAttribute {
    //Common
    ID("ID", String.class),
    EXTERNAL_ID("externalID", String.class),
    //CMS ID
    CMS_ID("cmsId", String.class),

    //Version
    VERSION("version", String.class),
    VERSION_STATUS("status", NuageVspApiVersion.Status.class),

    PARENT_ID("parentID", String.class),
    TEMPLATE_ID("templateID", String.class),
    APIKEY("APIKey", String.class),
    ENTERPRISE_ID("enterpriseID", String.class),
    LASTUPDATED("lastUpdatedDate", String.class),
    NAME("name", String.class),
    DESCRIPTION("description", String.class),

    //Enterprise
    ENTERPRISE_PROFILE_ID("enterpriseProfileID", String.class),

    //Group
    GROUP_PRIVATE("private", String.class),
    GROUP_ROLE("role", String.class),

    //User
    USER_USERNAME("userName", String.class),
    USER_FIRSTNAME("firstName", String.class),
    USER_LASTNAME("lastName", String.class),
    USER_PASSWORD("password", String.class),
    USER_EMAIL("email", String.class),

    //Domain Template

    //L2Domain Template
    L2DOMAIN_TEMPLATE_ADDRESS("address", String.class),
    L2DOMAIN_TEMPLATE_NETMASK("netmask", String.class),
    L2DOMAIN_TEMPLATE_GATEWAY("gateway", String.class),

    //Domain
    DOMAIN_PATENABLED("PATEnabled", Boolean.class),
    DOMAIN_UNDERLAYENABLED("underlayEnabled", Boolean.class),

    //Zone
    ZONE_PUBLIC("publicZone", String.class),

    //Subnet
    SUBNET_ADDRESS("address", String.class),
    SUBNET_NETMASK("netmask", String.class),
    SUBNET_GATEWAY("gateway", String.class),

    //Address Range
    ADDRESS_RANGE_MIN("minAddress", String.class),
    ADDRESS_RANGE_MAX("maxAddress", String.class),

    //L2Domain
    L2DOMAIN_ADDRESS("address", String.class),
    L2DOMAIN_NETMASK("netmask", String.class),
    L2DOMAIN_GATEWAY("gateway", String.class),

    DHCP_OPTIONS_LENGTH("length", String.class),
    DHCP_OPTIONS_TYPE("type", String.class),
    DHCP_OPTIONS_VALUE("value", String.class),
    DHCP_OPTIONS_ACTUAL_TYPE("actualType", Integer.class),
    DHCP_OPTIONS_ACTUAL_VALUES("actualValues", Types.LIST_OF_STRINGS),

    //ACL Template
    ACLTEMPLATES_ACTIVE("active", Boolean.class),
    ACLTEMPLATES_PRIORITY("priority", Integer.class),
    ACLTEMPLATES_PRIORITY_TYPE("priorityType", Acl.AclTemplatePriorityType.class),
    ACLTEMPLATES_POLICY_STATE("policyState", Acl.AclPolicyState.class),
    ACLTEMPLATES_ENTRIES("ACLEntries", NuageVspObjectTypeAdapter.LIST_TYPE),
    ACLTEMPLATES_ALLOW_NON_IP("defaultAllowNonIP", Boolean.class),
    ACLTEMPLATES_ALLOW_IP("defaultAllowIP", Boolean.class),

    //ACL Template Entries
    ACLTEMPLATES_ENTRY_LOCATION_TYPE("locationType", Acl.AclEntryLocationType.class),
    ACLTEMPLATES_ENTRY_NETWORK_TYPE("networkType", Acl.AclEntryNetworkType.class),
    ACLTEMPLATES_ENTRY_LOCATION_ID("locationID", String.class),
    ACLTEMPLATES_ENTRY_NETWORK_ID("networkID", String.class),
    ACLTEMPLATES_ENTRY_SOURCE_PORT("sourcePort", String.class),
    ACLTEMPLATES_ENTRY_DEST_PORT("destinationPort", String.class),
    ACLTEMPLATES_ENTRY_PROTOCOL("protocol", Protocol.class),
    ACLTEMPLATES_ENTRY_PRIORITY("priority", Integer.class),
    ACLTEMPLATES_ENTRY_ACTION("action", Acl.AclAction.class),
    ACLTEMPLATES_ENTRY_ETHER_TYPE("etherType", Acl.AclEtherType.class),
    ACLTEMPLATES_ENTRY_ADDR_OVERRIDE("addressOverride", String.class),
    ACLTEMPLATES_ENTRY_REFLEXIVE(ImmutableMap.of(NuageVspApiVersion.V3_2, "reflexive", NuageVspApiVersion.V4_0, "stateful")),
    ACLTEMPLATES_ENTRY_DSCP("DSCP", String.class),

    //FIP Rate Limiting QOS
    FIP_RATE_LIMITING_QOS_FIP_PEAK_INFORMATION_RATE("FIPPeakInformationRate", String.class),
    FIP_RATE_LIMITING_QOS_FIP_RATE_LIMITING_ACTIVE("FIPRateLimitingActive", String.class),
    FIP_RATE_LIMITING_QOS_FIP_PEAK_BURST_SIZE("FIPPeakBurstSize", String.class),
    FIP_RATE_LIMITING_QOS_ACTIVE("active", Boolean.class),

    //Virtual Machine
    VM_UUID("UUID", String.class),
    VM_STATUS("status", String.class),
    VM_INTERFACES("interfaces", NuageVspObjectTypeAdapter.LIST_TYPE),

    //VM Interface
    VM_INTERFACE_MAC("MAC", String.class),
    VM_INTERFACE_ATTACHED_NETWORK_ID("attachedNetworkID", String.class),
    VM_INTERFACE_ATTACHED_NETWORK_TYPE("attachedNetworkType", String.class),
    VM_INTERFACE_GATEWAY("gateway", String.class),
    VM_INTERFACE_IPADDRESS("IPAddress", String.class),
    VM_INTERFACE_NETMASK("netmask", String.class),
    VM_INTERFACE_VPORT_ID("VPortID", String.class),
    VM_INTERFACE_DOMAIN_ID("domainID", String.class),
    VM_INTERFACE_DOMAIN_NAME("domainName", String.class),
    VM_INTERFACE_ZONE_NAME("zoneName", String.class),
    VM_INTERFACE_NETWORK_NAME("networkName", String.class),
    VM_INTERFACE_VPORT_NAME("VPortName", String.class),
    VM_INTERFACE_ASSOCIATED_FIP("associatedFloatingIPAddress", String.class),
    VM_INTERFACE_VM_UUID("VMUUID", String.class),

    //Shared Resource
    SHARED_RESOURCE_ADRESS("address", String.class),
    SHARED_RESOURCE_GATEWAY("gateway", String.class),
    SHARED_RESOURCE_NETMASK("netmask", String.class),
    SHARED_RESOURCE_TYPE("type", NuageVspConstants.SharedNetworkType.class),
    SHARED_RESOURCE_UNDERLAY("underlay", Boolean.class),
    SHARED_RESOURCE_PARENT_ID("sharedResourceParentID", String.class),
    ASSOC_SHARED_NTWK_ID("associatedSharedNetworkResourceID", String.class),

    //FloatingIP
    FLOATING_IP_ADDRESS("address", String.class),
    FLOATING_IP_ASSIGNED("assigned", String.class),
    FLOATING_IP_ACCESS_CONTROL("accessControl", String.class),

    //VirtualIP
    VIRTUAL_IP_MAC("MAC", String.class),
    VIRTUAL_IP_ADDRESS("virtualIP", String.class),
    VIRTUAL_IP_FLOATING_IP_ID("associatedFloatingIPID", String.class),

    //VPort
    VPORT_TYPE("type", String.class),
    VPORT_ACTIVE("active", String.class),
    VPORT_ADDRESSSPOOFING("addressSpoofing", NuageVspConstants.AddressSpoofing.class),
    VPORT_FLOATING_IP_ID("associatedFloatingIPID", String.class),

    //Public network Macro
    ENTERPRISE_NTWK_MACRO_ADDRESS("address", String.class),
    ENTERPRISE_NTWK_MACRO_NETMASK("netmask", String.class),

    //Enterprise Profile
    ENTERPRISE_PROFILE_FLOATING_IP_QUOTA("floatingIPsQuota", String.class),
    ENTERPRISE_PROFILE_GATEWAY_MGMT("allowGatewayManagement", String.class),
    ENTERPRISE_PROFILE_FWD_CLASSES("allowedForwardingClasses", Types.LIST_OF_STRINGS),
    ENTERPRISE_PROFILE_ADV_QOS("allowAdvancedQOSConfiguration", String.class),

    //Policy Group
    POLICY_GROUP_TYPE("type", String.class),
    POLICY_GROUP_EXTERNAL("external", String.class),

    //Gateway
    GATEWAY_SYSTEMID("systemID", String.class),

    //Gateway Service
    WAN_SERVICE_VPN_CONNECT_ID("associatedVPNConnectID", String.class),

    //VPN connection
    VPN_CONNECTION_WANSERVICE_ID("associatedWANServiceID", String.class),

    //Static Route
    STATIC_ROUTE_ADDRESS("address", String.class),
    STATIC_ROUTE_NETMASK("netmask", String.class),
    STATIC_ROUTE_NEXTHOP("nextHopIp", String.class),

    //EnterprisePermission
    ENTERPRISEPERMISSION_PERMITTED_ENTITYID("permittedEntityID", String.class),
    ENTERPRISEPERMISSION_PERMITTED_ENTITYYPE("permittedEntityType", String.class),
    ENTERPRISEPERMISSION_PERMITTED_ACTION("permittedAction", String.class);

    //Cloud Management Systems

    private static Map<NuageVspApiVersion, Map<String, NuageVspAttribute>> lookup = new HashMap<NuageVspApiVersion, Map<String, NuageVspAttribute>>();

    static {
        for (NuageVspApiVersion apiVersion : NuageVspApiVersion.SUPPORTED_VERSIONS) {
            Map<String, NuageVspAttribute> attributes = new HashMap<String, NuageVspAttribute>();
            lookup.put(apiVersion, attributes);

            for (NuageVspAttribute attribute : NuageVspAttribute.values()) {
                attributes.put(attribute.getAttributeName(apiVersion), attribute);
            }
        }
    }

    private Type type;

    private Map<NuageVspApiVersion, String> attributeName = Maps.newHashMap();

    private NuageVspAttribute(String attributeName, Type type) {
        for (NuageVspApiVersion apiVersion : NuageVspApiVersion.SUPPORTED_VERSIONS) {
            this.attributeName.put(apiVersion, attributeName);
        }
        this.type = type;
    }

    private NuageVspAttribute(Map<NuageVspApiVersion, String> attributeNames, Type type) {
        if (attributeNames.size() > NuageVspApiVersion.SUPPORTED_VERSIONS.length) {
            throw new IllegalArgumentException("Can't specify more attribute names (" + attributeNames.size() + ") than there are supported " +
                    "versions (" + NuageVspApiVersion.SUPPORTED_VERSIONS.length + ")");
        }

        for (NuageVspApiVersion apiVersion : NuageVspApiVersion.SUPPORTED_VERSIONS) {
            if (!attributeNames.containsKey(apiVersion)) {
                throw new IllegalArgumentException("Please specify the attribute name for each supported version. Attribute " + this.name() + " is missing attribute name for " + apiVersion);
            }

            this.attributeName.put(apiVersion, attributeNames.get(apiVersion));
        }

        this.type = type;
    }


    private NuageVspAttribute(Map<NuageVspApiVersion, String> attributeNames) {
        this(attributeNames, String.class);
    }

    @Deprecated
    public String getAttributeName() {
        return attributeName.get(NuageVspApiVersion.CURRENT);
    }

    public String getAttributeName(NuageVspApiVersion apiVersion) {
        return attributeName.get(apiVersion);
    }

    public static NuageVspAttribute lookup(NuageVspApiVersion apiVersion, String attributeName, NuageVspEntity entityType) {
        for (NuageVspAttribute attr : entityType.getAttributes()) {
            if (attr.getAttributeName(apiVersion).equals(attributeName)) {
                return attr;
            }
        }
        return null;
    }

    @Override
    public String toString() {
        return getAttributeName(NuageVspApiVersion.CURRENT);
    }

    public Type getType() {
        return type;
    }

    private static class Types {
        public static final Type LIST_OF_STRINGS = new TypeToken<List<String>>() {}.getType();
    }
}
