/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.io.IOException;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import net.nuage.vsp.acs.client.common.FilterProcessor;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Function;

/**
 * Created by frmaximu on 19/10/2016.
 */
public class NuageVspFilter {

    interface FilterBuilderStrategy {
        void appendField(StringBuilder appendable, NuageVspAttribute attribute) throws IOException;
        void appendBinaryOperator(StringBuilder appendable, BinaryOp operator, Object value) throws IOException;
    }

    static class VspFilterBuilderStrategy implements FilterBuilderStrategy {
        public VspFilterBuilderStrategy(NuageVspApiVersion apiVersion) {
            this.apiVersion = apiVersion;
        }

        private NuageVspApiVersion apiVersion;

        private static final EnumMap<BinaryOp, String> operatorMap;

        static {
            operatorMap = new EnumMap<>(BinaryOp.class);
            operatorMap.put(BinaryOp.EQ, "==");
            operatorMap.put(BinaryOp.NE, "!=");
            operatorMap.put(BinaryOp.LT, "lt");
            operatorMap.put(BinaryOp.LE, "le");
            operatorMap.put(BinaryOp.GT, "gt");
            operatorMap.put(BinaryOp.GE, "ge");
            operatorMap.put(BinaryOp.LIKE, "LIKE");
            operatorMap.put(BinaryOp.CONTAINS, "CONTAINS");
            operatorMap.put(BinaryOp.STARTS_WITH, "BEGINSWITH");
            operatorMap.put(BinaryOp.ENDS_WITH, "ENDSWITH");
        }

        @Override
        public void appendField(StringBuilder appendable, NuageVspAttribute attribute) throws IOException {
            appendable.append(attribute.getAttributeName(apiVersion));
        }

        @Override
        public void appendBinaryOperator(StringBuilder appendable, BinaryOp operator, Object value)  throws IOException {
            appendable
                    .append(operatorMap.get(operator))
                    .append(" ");

             if (value instanceof CharSequence) {
                 appendable.append("'");
                 appendable.append(value.toString());
                 appendable.append("'");
             } else if (value != null) {
                 appendable.append(value.toString());
             } else {
                 appendable.append("null");
             }
        }
    }

    private FilterBuilderStrategy filterBuilderStrategy = new VspFilterBuilderStrategy(NuageVspApiVersion.CURRENT);

    private Filter filter;
    private LinkedList<Filter> filterStack = new LinkedList<>();
    private LinkedList<OrderBy> orderByList = new LinkedList<>();

    private enum UnaryOp { NOT }
    protected enum BinaryOp { EQ, NE, LT, GT, LE, GE, LIKE, CONTAINS, STARTS_WITH, ENDS_WITH}
    private enum GroupOp { OR, AND }
    public enum OrderType { ASC, DESC }

    private interface Filter {
        void appendFilter(StringBuilder sb) throws IOException;
        default void addFilter(Filter filter) {
            throw new UnsupportedOperationException();
        }
        default boolean accept(GroupOp op) {
            return false;
        }
        void applyCmsId(String cmsId);
    }

    private class OrderBy {
        private NuageVspAttribute attribute;
        private OrderType type;

        public OrderBy(NuageVspAttribute attribute, OrderType type) {
            this.attribute = attribute;
            this.type = type;
        }

        public NuageVspAttribute getAttribute() {
            return attribute;
        }

        public OrderType getType() {
            return type;
        }

        public void appendOrderBy(StringBuilder appendable) throws IOException{
            filterBuilderStrategy.appendField(appendable, attribute);
            appendable.append(" ").append(type.name());
        }
    }

    private static class LegacyFilter implements Filter {
        private String filter;

        private LegacyFilter(String filter) {
            this.filter = filter;
        }

        public void appendFilter(StringBuilder sb) throws IOException {
            sb.append(filter);
        }

        @Override
        public void applyCmsId(String cmsId) {
            filter = FilterProcessor.processFilter(filter, cmsId);
        }
    }

    private class UnaryOperator implements Filter {
        UnaryOp operator;
        Filter filter;

        public UnaryOperator(UnaryOp operator, Filter filter) {
            this.operator = operator;
            this.filter = filter;
        }

        public void addFilter(Filter filter) {
            this.filter = filter;
        }

        @Override
        public void applyCmsId(String cmsId) {
            filter.applyCmsId(cmsId);
        }

        @Override
        public void appendFilter (StringBuilder sb) throws IOException {
            switch (operator) {
                case NOT:
                    sb.append("NOT");
                    if (!(filter instanceof GroupOperator)) {
                        sb.append("(");
                    }
                    filter.appendFilter(sb);
                    if (!(filter instanceof GroupOperator)) {
                        sb.append(")");
                    }
                    break;
            }
        }
    }

    private class GroupOperator implements Filter {
        GroupOp operator;
        List<Filter> filters;
        boolean parantheses;

        public GroupOperator(GroupOp operator, List<Filter> filters) {
            this.operator = operator;
            this.filters = new LinkedList<>(filters);
        }

        public GroupOperator(GroupOp operator, Filter filter, boolean parantheses) {
            this.operator = operator;
            this.filters = new LinkedList<>();
            this.filters.add(filter);
            this.parantheses = parantheses;
        }

        public void addFilter(Filter filter) {
            filters.add(filter);
        }

        public void withParantheses() {
            parantheses = true;
        }

        @Override
        public void applyCmsId(String cmsId) {
            for (Filter filter : filters) {
                filter.applyCmsId(cmsId);
            }
        }

        @Override
        public void appendFilter (StringBuilder sb) throws IOException {
            Iterator<Filter> parts = filters.iterator();
            if (parantheses) {
                sb.append("(");
            }

            if(parts.hasNext()) {
                parts.next().appendFilter(sb);
                while(parts.hasNext()) {
                    sb.append(" ").append(operator.name()).append(" ");
                    parts.next().appendFilter(sb);
                }
            }

            if (parantheses) {
                sb.append(")");
            }
        }

        @Override
        public boolean accept(GroupOp op) {
            return op == operator;
        }
    }

    private class BinaryOperator implements Filter {
        NuageVspAttribute attribute;
        BinaryOp operator;
        Object value;

        public BinaryOperator(NuageVspAttribute attribute, BinaryOp operator, Object value) {
            this.attribute = attribute;
            this.operator = operator;
            this.value = value;
        }

        @Override
        public void appendFilter (StringBuilder sb) throws IOException {
            filterBuilderStrategy.appendField(sb, attribute);
            sb.append(" ");
            filterBuilderStrategy.appendBinaryOperator(sb, operator, value);
        }

        @Override
        public void applyCmsId(String cmsId) {
            if (FilterProcessor.EXTERNAL_ID_FIELDS.contains(attribute) && operator == BinaryOp.EQ) {
                String externalId = (String) value;

                if (externalId.contains(NuageVspConstants.EXTERNAL_ID_DELIMITER)) {
                    final String oldCmsId = StringUtils.substringAfter(externalId, NuageVspConstants.EXTERNAL_ID_DELIMITER);
                    if (oldCmsId.equals(cmsId)) {
                        return;
                    } else {
                        value = StringUtils.substringBefore(externalId, NuageVspConstants.EXTERNAL_ID_DELIMITER);
                    }
                }

                value = value + NuageVspConstants.EXTERNAL_ID_DELIMITER + cmsId;
            }
        }
    }

    public class BinaryOperatorBuilder {
        NuageVspAttribute attribute;

        private BinaryOperatorBuilder(NuageVspAttribute attribute) {
            this.attribute = attribute;
        }

        public NuageVspFilter eq(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.EQ, value));
        }

        public NuageVspFilter gt(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.GT, value));
        }

        public NuageVspFilter lt(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.LT, value));
        }

        public NuageVspFilter ge(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.GE, value));
        }

        public NuageVspFilter le(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.LE, value));
        }

        public NuageVspFilter ne(Object value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.NE, value));
        }

        public NuageVspFilter startsWith(String value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.STARTS_WITH, value));
        }

        public NuageVspFilter endsWith(String value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.ENDS_WITH, value));
        }

        public NuageVspFilter contains(String value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.CONTAINS, value));
        }

        public NuageVspFilter like(String value) {
            return NuageVspFilter.this.addFilter(new BinaryOperator(attribute, BinaryOp.LIKE, value));
        }
    }

    private NuageVspFilter addFilter(Filter aFilter) {
        if (filter != null) {
            this.filter.addFilter(aFilter);
        } else {
            this.filter = aFilter;
        }
        return this;
    }

    public static NuageVspFilter where() {
        return new NuageVspFilter();
    }

    public NuageVspFilter orderBy(NuageVspAttribute attribute, OrderType type) {
        orderByList.add(new OrderBy(attribute, type));
        return this;
    }

    @Deprecated
    public static NuageVspFilter where(String legacyFilter) {
        NuageVspFilter f =  new NuageVspFilter();
        f.filter = new LegacyFilter(legacyFilter);
        return f;
    }

    public static BinaryOperatorBuilder where(NuageVspAttribute attribute) {
        return where().field(attribute);
    }

    public NuageVspFilter not(Function<NuageVspFilter, NuageVspFilter> consumer) {
        filterStack.push(filter);
        filter = null;
        consumer.apply(this);
        Filter currentFilter = filter;
        filter = filterStack.pop();
        addFilter(new UnaryOperator(UnaryOp.NOT, currentFilter));
        return this;
    }

    private NuageVspFilter startGroup(GroupOp op) {
        if (filter != null && !filter.accept(op)) {
            this.filter = new GroupOperator(op, this.filter, !filterStack.isEmpty());
        }
        return this;
    }

    public BinaryOperatorBuilder and(NuageVspAttribute attribute) {
        return and().field(attribute);
    }

    public NuageVspFilter and() {
        return startGroup(GroupOp.AND);
    }

    public BinaryOperatorBuilder or(NuageVspAttribute attribute) {
        return or().field(attribute);
    }

    public NuageVspFilter or() {
        return startGroup(GroupOp.OR);
    }

    public BinaryOperatorBuilder field(NuageVspAttribute attribute) {
        return this.new BinaryOperatorBuilder(attribute);
    }

    public NuageVspFilter op() {
        filterStack.push(filter);
        filter = null;
        return this;
    }

    public NuageVspFilter cp() {
        Filter currentFilter = filter;
        filter = filterStack.pop();
        addFilter(currentFilter);
        return this;
    }


    public void applyCmsId(String cmsId) {
        if (filter != null) {
            filter.applyCmsId(cmsId);
        }
    }

    public String toString() {
        final StringBuilder sb = new StringBuilder();
        try {
            filter.appendFilter(sb);
        } catch (IOException ignored) {}
        return sb.toString();
    }

    public String getFilterString(NuageVspApiVersion apiVersion) {
        filterBuilderStrategy = new VspFilterBuilderStrategy(apiVersion);
        return toString();
    }

    public String getFilterString(FilterBuilderStrategy strategy) {
        filterBuilderStrategy = strategy;
        return toString();
    }


    public boolean hasOrderBy() {
        return !orderByList.isEmpty();
    }

    public String getOrderByString(FilterBuilderStrategy strategy) {
        filterBuilderStrategy = strategy;
        return getOrderByString();
    }

    public String getOrderByString() {
        if (orderByList.isEmpty()) {
            return null;
        }

        final StringBuilder sb = new StringBuilder();
        try {
            Iterator<OrderBy> orderBys = orderByList.iterator();
            if(orderBys.hasNext()) {
                orderBys.next().appendOrderBy(sb);
                while(orderBys.hasNext()) {
                    sb.append(", ");
                    orderBys.next().appendOrderBy(sb);
                }
            }
        } catch (IOException ignored) {}
        return sb.toString();
    }
}
