/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.io.Serializable;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import net.nuage.vsp.acs.client.common.NuageVspApiVersion;

import org.apache.commons.lang.ObjectUtils;

import com.google.common.base.Function;

public class NuageVspObject implements Serializable {
    private final EnumMap<NuageVspAttribute, Object> data;
    private final NuageVspEntity entity;
    private final NuageVspApiVersion apiVersion;

    private final Function<NuageVspAttribute, String> getAttributeValueFunction = new Function<NuageVspAttribute, String>() {
        @Override
        public String apply(NuageVspAttribute attribute) {
            return get(attribute);
        }
    };

    public NuageVspObject(NuageVspApiVersion apiVersion, NuageVspEntity entity) {
        this.data = new EnumMap<>(NuageVspAttribute.class);
        this.apiVersion = apiVersion;
        this.entity = entity;
    }


    public NuageVspObject copy() {
        NuageVspObject copy = new NuageVspObject(this.apiVersion, this.entity);
        copy.data.putAll(this.data);
        return copy;
    }

    public NuageVspObject(NuageVspApiVersion apiVersion, NuageVspEntity entity, String id) {
        this(apiVersion, entity);
        setId(id);
    }

    @SuppressWarnings("unchecked")
    public <T> T get(NuageVspAttribute field) {
        assert entity.hasAttribute(field);
        return (T) data.get(field);
    }

    public NuageVspObject set(NuageVspAttribute field, Object value) {
        assert entity.hasAttribute(field);

        data.put(field, value);
        return this;
    }

    public NuageVspObject unset(NuageVspAttribute field) {
        assert entity.hasAttribute(field);

        data.remove(field);
        return this;
    }

    public boolean hasAttribute(NuageVspAttribute field) {
        return data.containsKey(field);
    }

    public NuageVspApiVersion getApiVersion() {
        return apiVersion;
    }

    public NuageVspEntity getEntityType() {
        return entity;
    }

    public NuageVspObject setExternalId(String value) {
        return set(NuageVspAttribute.EXTERNAL_ID, value);
    }

    public void setId(String value) {
        set(NuageVspAttribute.ID, value);
    }

    public String getId() {
        return get(NuageVspAttribute.ID);
    }

    public String getExternalId() {
        return get(NuageVspAttribute.EXTERNAL_ID);
    }

    public String getName() {
        return entity.getNameAttribute().transform(getAttributeValueFunction).orNull();
    }

    public String getDescription() {
        try {
            return entity.getDescriptionAttribute().transform(getAttributeValueFunction).orNull();
        } catch (NullPointerException npe) {
            return null;
        }
    }

    public boolean entriesEqualTo(Object o) {
        if (o == null || !(o instanceof NuageVspObject)) return false;
        NuageVspObject other = (NuageVspObject) o;
        if (!entity.equals(other.getEntityType())) return false;
        for (Map.Entry<NuageVspAttribute, Object> entry : other.data.entrySet()) {
            Object thisValue = this.data.get(entry.getKey());
            Object thatValue = entry.getValue();
            if (!ObjectUtils.equals(thisValue, thatValue)) return false;
        }
        return true;
    }

    public Set<Map.Entry<NuageVspAttribute, Object>> entrySet() {
        return data.entrySet();
    }

    @Override public String toString() {
        return data.toString();
    }

    public String toFilterString() {
        StringBuilder appendable = new StringBuilder();
        final Iterator<Map.Entry<NuageVspAttribute, Object>> entryIterator = data.entrySet().iterator();
        if (entryIterator.hasNext()) {
            appendFilterString(appendable, entryIterator.next());
            while (entryIterator.hasNext()) {
                appendable.append(" AND ");
                appendFilterString(appendable, entryIterator.next());
            }
        }
        return appendable.toString();
    }

    private void appendFilterString(StringBuilder appendable, Map.Entry<NuageVspAttribute, Object> entry) {
        String operator = (entry.getKey() == NuageVspAttribute.EXTERNAL_ID) ? " BEGINSWITH " :  " == ";
        appendable.append(entry.getKey().getAttributeName(apiVersion));
        appendable.append(operator);
        Object value = entry.getValue();
        if (value instanceof CharSequence) {
            appendable.append('"').append((CharSequence)value).append('"');
        } else {
            appendable.append(value.toString());
        }
    }

    @Override public boolean equals(Object obj) {
        if (obj == null || !(obj instanceof NuageVspObject)) return false;
        NuageVspObject that = (NuageVspObject) obj;

        if (this == that) {
            return true;
        }

        return Objects.equals(this.entity, that.entity)
                && Objects.equals(data, that.data);
    }

    @Override public int hashCode() {
        return data.hashCode();
    }
}
