/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model.gson;

import java.lang.reflect.Type;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import net.nuage.vsp.acs.client.common.FilterProcessor;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;

@SuppressWarnings("rawtypes")
public class NuageVspObjectTypeAdapter implements JsonDeserializer<NuageVspObject>, JsonSerializer<NuageVspObject> {
    public static final Type LIST_TYPE = new TypeToken<List<NuageVspObject>>() {}.getType();

    private NuageVspEntity entity;
    private NuageVspApiVersion apiVersion;
    private String externalIdSuffix;
    private String cmsId;

    public NuageVspObjectTypeAdapter(NuageVspEntity entity, NuageVspApiVersion apiVersion, String cmsId) {
        this.entity = entity;
        this.apiVersion = apiVersion;
        this.cmsId = cmsId;
        this.externalIdSuffix = NuageVspConstants.EXTERNAL_ID_DELIMITER + cmsId;
    }

    public void setEntity(NuageVspEntity entity) {
        this.entity = entity;
    }

    @Override
    public NuageVspObject deserialize(JsonElement src, Type srcType, JsonDeserializationContext context) throws JsonParseException {
        return deserialize(entity, src, context);
    }

    private NuageVspObject deserialize(NuageVspEntity entity, JsonElement src, JsonDeserializationContext context) throws JsonParseException {
        NuageVspObject obj = new NuageVspObject(apiVersion, entity);
        JsonObject json = src.getAsJsonObject();

        for (Entry<String, JsonElement> entry : json.entrySet()) {
            NuageVspAttribute attribute = NuageVspAttribute.lookup(apiVersion, entry.getKey(), entity);
            if(attribute == null){
                continue;
            }
            final JsonElement jsonValue = entry.getValue();
            if (jsonValue.isJsonNull()) {
                continue;
            }

            Object value = null;

            if (jsonValue.isJsonPrimitive()) {
                value = deserializePrimitive(context, obj, attribute, jsonValue);

                if (value == null) {
                    continue;
                }
            } else if (jsonValue.isJsonObject()) {
                value = context.deserialize(jsonValue, NuageVspObject.class);
            } else if (jsonValue.isJsonArray()) {
                if (attribute.getType().equals(LIST_TYPE)) {
                    value = deserializeChild(entity, context, jsonValue);
                } else {
                    value = context.deserialize(jsonValue, List.class);
                }
            }

            if (value != null) {
                obj.set(attribute, value);
            }
        }

        return obj;
    }

    private Object deserializePrimitive(JsonDeserializationContext context, NuageVspObject obj, NuageVspAttribute attribute, JsonElement jsonValue) {
        Object value = null;
        JsonPrimitive primitive = jsonValue.getAsJsonPrimitive();
        if (primitive.isBoolean()) {
            value = primitive.getAsBoolean();
        } else if (primitive.isNumber()) {
            Number n = jsonValue.getAsNumber();
            if (attribute.getType() == Integer.class) {
                value = n.intValue();
            } else if (attribute.getType() == Long.class) {
                value = n.longValue();
            } else {
                value = n;
            }
        } else if (primitive.isString()) {
            final String primitiveAsString = primitive.getAsString();

            if (FilterProcessor.EXTERNAL_ID_FIELDS.contains(attribute) && primitiveAsString.endsWith(externalIdSuffix))  {
                value = primitiveAsString.substring(0, primitiveAsString.length() - externalIdSuffix.length());
                obj.set(NuageVspAttribute.CMS_ID, cmsId);
            } else {
                value = context.deserialize(jsonValue, attribute.getType());
            }

        }
        return value;
    }

    private Object deserializeChild(NuageVspEntity entity, JsonDeserializationContext context, JsonElement jsonValue) {
        Object value;
        NuageVspEntity childEntity;
        switch(entity) {
            case EGRESS_ACLTEMPLATES: childEntity = NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES; break;
            case EGRESS_DOMAIN_FIP_ACLTEMPLATES: childEntity = NuageVspEntity.EGRESS_DOMAIN_FIP_ACLTEMPLATES_ENTRIES; break;
            case INGRESS_ACLTEMPLATES: childEntity = NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES; break;
            case VM: childEntity = NuageVspEntity.VM_INTERFACE; break;
            default: throw new JsonParseException("Unexpected Child entity");
        }

        List<NuageVspObject> elements = new LinkedList<>();
        for (JsonElement element : jsonValue.getAsJsonArray()) {
            elements.add(deserialize(childEntity, element, context));
        }
        value = elements;
        return value;
    }

    @Override public JsonElement serialize(NuageVspObject entity, Type typeOfSrc, JsonSerializationContext context) {
        JsonObject jsonObject = new JsonObject();
        for (NuageVspAttribute attribute : entity.getEntityType().getAttributes()) {
            if (entity.hasAttribute(attribute)) {
                JsonElement value;
                final Object o = entity.get(attribute);
                if (FilterProcessor.EXTERNAL_ID_FIELDS.contains(attribute))  {
                    value = new JsonPrimitive(o.toString() + externalIdSuffix);
                } else {
                    value =  context.serialize(o);
                }
                jsonObject.add(attribute.getAttributeName(apiVersion), value);
            }
        }
        return jsonObject;
    }
}
