/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.utils;

import java.lang.reflect.Array;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.net.util.SubnetUtils;

/**
 * Partial copy of the CloudStack NetUtils
 */
public class NetUtils {

    private static final int MAX_CIDR = 32;

    public static boolean isValidIp(final String ip) {
        if (ip.endsWith(".")) {
            return false;
        }

        final String[] ipAsList = ip.split("\\.");

        // The IP address must have four octets
        if (Array.getLength(ipAsList) != 4) {
            return false;
        }

        for (int i = 0; i < 4; i++) {
            // Each octet must be an integer
            final String octetString = ipAsList[i];
            int octet;
            try {
                octet = Integer.parseInt(octetString);
            } catch (final Exception e) {
                return false;
            }
            // Each octet must be between 0 and 255, inclusive
            if (octet < 0 || octet > 255) {
                return false;
            }

            // Each octetString must have between 1 and 3 characters
            if (octetString.length() < 1 || octetString.length() > 3) {
                return false;
            }
        }

        // IP is good, return true
        return true;
    }

    public static boolean isValidCIDR(final String cidr) {
        if (cidr == null || cidr.isEmpty()) {
            return false;
        }
        final String[] cidrPair = cidr.split("\\/");
        if (cidrPair.length != 2) {
            return false;
        }
        final String cidrAddress = cidrPair[0];
        final String cidrSize = cidrPair[1];
        if (!isValidIp(cidrAddress)) {
            return false;
        }
        int cidrSizeNum = -1;

        try {
            cidrSizeNum = Integer.parseInt(cidrSize);
        } catch (final Exception e) {
            return false;
        }

        if (cidrSizeNum < 0 || cidrSizeNum > MAX_CIDR) {
            return false;
        }

        return true;
    }

    public static Long[] cidrToLong(String cidr) {
        if (cidr == null || cidr.isEmpty()) {
            return null;
        }
        String[] cidrPair = cidr.split("\\/");
        if (cidrPair.length != 2) {
            return null;
        }
        String cidrAddress = cidrPair[0];
        String cidrSize = cidrPair[1];
        if (!isValidIp(cidrAddress)) {
            return null;
        }
        int cidrSizeNum = -1;

        try {
            cidrSizeNum = Integer.parseInt(cidrSize);
        } catch (Exception e) {
            return null;
        }
        long numericNetmask = (0xffffffff >> (32 - cidrSizeNum)) << (32 - cidrSizeNum);
        long ipAddr = ip2Long(cidrAddress);
        Long[] cidrlong = {ipAddr & numericNetmask, (long)cidrSizeNum};
        return cidrlong;

    }

    public static String getCidrSubNet(String cidr) {
        if (cidr == null || cidr.isEmpty()) {
            return null;
        }
        String[] cidrPair = cidr.split("/");
        if (cidrPair.length != 2) {
            return null;
        }
        String cidrAddress = cidrPair[0];
        String cidrSize = cidrPair[1];
        if (!isValidIp(cidrAddress)) {
            return null;
        }
        int cidrSizeNum;

        try {
            cidrSizeNum = Integer.parseInt(cidrSize);
        } catch (Exception e) {
            return null;
        }
        long numericNetmask = (0xffffffff >> (32 - cidrSizeNum)) << (32 - cidrSizeNum);
        String netmask = NetUtils.long2Ip(numericNetmask);
        return getSubNet(cidrAddress, netmask);
    }

    public static String getCidrNetmask(long cidrSize) {
        long numericNetmask = (0xffffffff >> (32 - cidrSize)) << (32 - cidrSize);
        return long2Ip(numericNetmask);
    }

    public static String getCidrNetmask(String cidr) {
        String[] cidrPair = cidr.split("/");
        long guestCidrSize = Long.parseLong(cidrPair[1]);
        return getCidrNetmask(guestCidrSize);
    }

    public static Pair<String, String> cidrToSubnetNetmask(String sourceCidr) {
        String netmask;
        String subnet = getCidrSubNet(sourceCidr);
        long cidrSize = new Long(sourceCidr.substring(sourceCidr.indexOf("/") + 1));
        if (cidrSize == 0) {
            netmask = "0.0.0.0";
        } else {
            netmask = getCidrNetmask(sourceCidr);
        }

        return Pair.of(subnet, netmask);
    }

    public static String long2Ip(long ip) {
        return String.valueOf((ip >> 24 & 0xff)) + "." + (ip >> 16 & 0xff) + "." + (ip >> 8 & 0xff) + "." + (ip & 0xff);
    }

    public static String getSubNet(String ip, String netmask) {
        long ipAddr = ip2Long(ip);
        long subnet = ip2Long(netmask);
        long result = ipAddr & subnet;
        return long2Ip(result);
    }

    public static String getCidrFromGatewayAndNetmask(String gatewayStr, String netmaskStr) {
        long netmask = ip2Long(netmaskStr);
        long gateway = ip2Long(gatewayStr);
        long firstPart = gateway & netmask;
        long size = getCidrSize(netmaskStr);
        return long2Ip(firstPart) + "/" + size;
    }

    public static long getCidrSize(String netmask) {
        long ip = ip2Long(netmask);
        int count = 0;
        for (int i = 0; i < 32; i++) {
            if (((ip >> i) & 0x1) == 0) {
                count++;
            } else {
                break;
            }
        }

        return 32 - count;
    }

    public static long ip2Long(String ip) {
        String[] tokens = ip.split("[.]");
        assert (tokens.length == 4);
        long result = 0;
        for (String token : tokens) {
            try {
                result = (result << 8) | Integer.parseInt(token);
            } catch (NumberFormatException e) {
                throw new RuntimeException("Incorrect number", e);
            }
        }

        return result;
    }

    public static boolean isIpWithinCidrRange(final String ipAddress, final String cidr) {
        if (!isValidIp(ipAddress)) {
            return false;
        }
        if (!isValidCIDR(cidr)) {
            return false;
        }

        // check if the gatewayip is the part of the ip range being added.
        // RFC 3021 - 31-Bit Prefixes on IPv4 Point-to-Point Links
        //     GW              Netmask         Stat IP        End IP
        // 192.168.24.0 - 255.255.255.254 - 192.168.24.0 - 192.168.24.1
        // https://tools.ietf.org/html/rfc3021
        // Added by Wilder Rodrigues
        final SubnetUtils subnetUtils = new SubnetUtils(cidr);
        subnetUtils.setInclusiveHostCount(true);

        final boolean isInRange = subnetUtils.getInfo().isInRange(ipAddress);

        return isInRange;
    }

}
