/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.function.Supplier;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.NuageVspApiClient;
import net.nuage.vsp.acs.client.api.model.NetworkRelatedVsdIds;
import net.nuage.vsp.acs.client.api.model.VspAddressRange;
import net.nuage.vsp.acs.client.api.model.VspDhcpDomainOption;
import net.nuage.vsp.acs.client.api.model.VspDomain;
import net.nuage.vsp.acs.client.api.model.VspNetwork;
import net.nuage.vsp.acs.client.api.model.VspNic;
import net.nuage.vsp.acs.client.api.model.VspStaticNat;
import net.nuage.vsp.acs.client.api.model.VspVm;
import net.nuage.vsp.acs.client.common.ConfigUtil;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.Acl;
import net.nuage.vsp.acs.client.common.model.Dhcp;
import net.nuage.vsp.acs.client.common.model.DhcpOption;
import net.nuage.vsp.acs.client.common.model.DhcpOptions;
import net.nuage.vsp.acs.client.common.model.NetworkDetails;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.common.utils.Logger;
import net.nuage.vsp.acs.client.common.utils.NetUtils;
import net.nuage.vsp.acs.client.common.utils.UuidUtils;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspException;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.net.util.SubnetUtils;

import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import static net.nuage.vsp.acs.client.common.NuageVspConstants.SharedNetworkType;
import static net.nuage.vsp.acs.client.common.model.NuageVspFilter.where;

public class NuageVspApiClientImpl implements NuageVspApiClient {

    private static final Logger s_logger = new Logger(NuageVspApiClientImpl.class);
    protected final NuageVspRestApi api;

    private NuageVspAclClient aclClient;

    public NuageVspApiClientImpl(NuageVspRestApi api, NuageVspAclClient aclClient) {
        this.api = api;
        this.aclClient = aclClient;
    }

    @Override public boolean entityExists(NuageVspEntity entityType, String entityUuid) throws NuageVspException {
        NuageVspObject entity = api.getEntityByExternalId(entityType, null, null, entityUuid);
        return entity != null;
    }

    public List<NuageVspObject> createVMInVSP(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, NetworkDetails attachedNetworkDetails, DhcpOptions dhcpOptions)
            throws NuageVspException {

        s_logger.debug("VM with UUID " + vspVm.getUuid() + " does not exist in VSP. So, just add the new VM");
        List<NuageVspObject> vmInterfacesList = buildVmInterfacesList(vspVm, vspNic);
        createVportInVsp(vspNic, vspNetwork, vmInterfacesList, attachedNetworkDetails, dhcpOptions);

        NuageVspObject vmEntity = api.createNuageVspObject(NuageVspEntity.VM);
        vmEntity.set(NuageVspAttribute.NAME, vspVm.getName());
        vmEntity.set(NuageVspAttribute.VM_UUID, vspVm.getUuid());
        vmEntity.set(NuageVspAttribute.VM_INTERFACES, vmInterfacesList);
        vmEntity.setExternalId(vspVm.getUuid());

        try {
            NuageVspObject vm = api.createResource(vmEntity, vspNetwork.getAccountUuid(), vspNetwork.getVspDomain()
                                                                                                    .getUuid());
            s_logger.debug("Created VM in Nuage. Response from VSP is " + vm);
            return vm.get(NuageVspAttribute.VM_INTERFACES);
        } catch (NuageVspApiException e) {
            throw NuageVspRestApi.handleException("Failed to create VM in VSP using REST API", e);
        }
    }

    public List<NuageVspObject> addVMInterfaceToVM(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, NetworkDetails attachedNetworkDetails, NuageVspObject vm,
            DhcpOptions dhcpOptions) throws NuageVspException {

        s_logger.debug("VM with UUID " + vspVm.getUuid() + " already exists in VSP. So, just add the VM new VM interface");

        String vmId = vm.get(NuageVspAttribute.ID);
        List<NuageVspObject> vmInterfacesFromNuageVSP = vm.get(NuageVspAttribute.VM_INTERFACES);
        NuageVspObject vmInterface = findVMInterface(vmInterfacesFromNuageVSP, vspNic.getMacAddress());
        if (vmInterface == null) {
            try {
                List<NuageVspObject> vmInterfacesList = buildVmInterfacesList(vspVm, vspNic);
                createVportInVsp(vspNic, vspNetwork, vmInterfacesList, attachedNetworkDetails, dhcpOptions);
                //Execute VMInterface creation as Account User
                NuageVspObject vmInterfaceJson = api.createResource(NuageVspEntity.VM, vmId, vmInterfacesList.get(0), vspNetwork.getAccountUuid(), vspNetwork.getVspDomain()
                                                                                                                                                             .getUuid());
                s_logger.debug("Added VM interface to VM in Nuage. Response from VSP is " + vmInterfaceJson);
                return Arrays.asList(vmInterfaceJson);
            } catch (NuageVspApiException exception) {
                throw NuageVspRestApi.handleException("Failed to add VM Interface for the VM with UUID %s for network %s. " + "Json response from VSP REST API is  %s", exception,
                                                      vspVm.getUuid(), vspNetwork.getUuid(), exception.getMessage());

            }
        } else {
            String vmInterfaceIp = vmInterface.get(NuageVspAttribute.VM_INTERFACE_IPADDRESS);
            if (!StringUtils.equals(vmInterfaceIp, vspNic.getIp())) {
                try {
                    if (vspNetwork.isShared()) {
                        String vportId = vmInterface.get(NuageVspAttribute.VM_INTERFACE_VPORT_ID);
                        String vmInterfaceUuid = vmInterface.getExternalId();

                        if (vspNetwork.isPublicAccess()) {
                            releaseFIPFromVsp(attachedNetworkDetails, vportId, vmInterfaceUuid, null);
                        }

                        boolean subnetMove = !NetUtils.isIpWithinCidrRange(vmInterfaceIp, vspNetwork.getCidr());
                        if (subnetMove) {
                            NuageVspObject vmInterfaceJson = moveVport(vspNetwork, vspVm, vspNic, attachedNetworkDetails, dhcpOptions, vmId, vmInterface, vportId);
                            return Arrays.asList(vmInterfaceJson);
                        } else if (vspNetwork.isPublicAccess()) {
                            makeInterfacePubliclyReachable(vspNetwork, vspNic.getIp(), attachedNetworkDetails, vportId, vmInterfaceUuid);
                        }
                    }

                    String vmInterfaceId = vmInterface.get(NuageVspAttribute.ID);
                    NuageVspObject updatedVmInterface = api.createNuageVspObject(NuageVspEntity.VM_INTERFACE);
                    updatedVmInterface.setId(vmInterfaceId);
                    updatedVmInterface.set(NuageVspAttribute.VM_INTERFACE_IPADDRESS, vspNic.getIp());
                    api.updateResource(updatedVmInterface);
                    s_logger.debug("Updated VM interface IP from " + vmInterfaceIp + " to " + vspNic.getIp());
                } catch (NuageVspApiException exception) {
                    throw NuageVspRestApi.handleException("Failed to update VM Interface with a new IP for the VM with UUID %s for network %s", exception, vspVm.getUuid(),
                                                          vspNetwork.getUuid());
                }
            }

            if (dhcpOptions != null) {
                // We still need to update the dhcpOptions if changed
                String vportVsdId = findEntityIdByExternalUuid(attachedNetworkDetails.getSubnetType(), attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT, vspNic.getUuid());
                createDhcpOptions(false, NuageVspEntity.VPORT, vportVsdId, vspNic.getUuid(), vspNetwork, dhcpOptions);
            }
        }

        return null;
    }

    private void makeInterfacePubliclyReachable(VspNetwork vspNetwork, String nicIp, NetworkDetails attachedNetworkDetails, String vportId, String vmInterfaceUuid)
            throws NuageVspException {
        String netmask = NetUtils.getCidrNetmask(vspNetwork.getCidr());
        String externalIdFromGateway = vspNetwork.getSubnetExternalId();
        String sharedResourceJson = findSharedResource(true, SharedNetworkType.FLOATING, externalIdFromGateway, vspNetwork.getGateway(), netmask);
        if (StringUtils.isBlank(sharedResourceJson)) {
            String errorMessage = "Failed to find the Floating IP subnet related to shared network " + vspNetwork.getUuid() + ".";
            s_logger.error(errorMessage);
            throw new NuageVspException(errorMessage);
        }

        String sharedResourceId = getEntityId(sharedResourceJson, NuageVspEntity.SHARED_NETWORK);
        allocateFIPToVPortInVsp(attachedNetworkDetails, vportId, sharedResourceId, nicIp, vmInterfaceUuid, null, null);
    }

    private NuageVspObject moveVport(VspNetwork vspNetwork, VspVm vspVm, VspNic vspNic, NetworkDetails attachedNetworkDetails, DhcpOptions dhcpOptions, String vmId,
            NuageVspObject vmInterface, String vportId) throws NuageVspException {
        api.deleteQuietly(NuageVspEntity.VM_INTERFACE, vmInterface.<String>get(NuageVspAttribute.ID));
        api.deleteQuietly(NuageVspEntity.VPORT, vportId);
        List<NuageVspObject> vmInterfacesList = buildVmInterfacesList(vspVm, vspNic);
        createVportInVsp(vspNic, vspNetwork, vmInterfacesList, attachedNetworkDetails, dhcpOptions);
        NuageVspObject vmInterfaceJson = api.createResource(NuageVspEntity.VM, vmId, vmInterfacesList.get(0), vspNetwork.getAccountUuid(), vspNetwork.getVspDomain()
                                                                                                                                                     .getUuid());
        s_logger.debug("Added VM interface to VM in Nuage. Response from VSP is " + vmInterfaceJson);
        return vmInterfaceJson;
    }

    @Override public List<NuageVspObject> buildVmInterfacesList(VspVm vspVm, VspNic... vspNics) {
        List<NuageVspObject> vmInterfaceList = new ArrayList<NuageVspObject>(vspNics.length);
        for (VspNic vspNic : vspNics) {
            NuageVspObject vmInterfaces = api.createNuageVspObject(NuageVspEntity.VM_INTERFACE);
            vmInterfaces.set(NuageVspAttribute.NAME, vspNic.getUuid());
            vmInterfaces.set(NuageVspAttribute.VM_INTERFACE_MAC, vspNic.getMacAddress());
            vmInterfaces.setExternalId(vspNic.getUuid());
            //If VM is a Virtual Router then set the static that was reserved earlier
            if (vspVm.getDomainRouter() == Boolean.TRUE) {
                vmInterfaces.set(NuageVspAttribute.VM_INTERFACE_IPADDRESS, vspVm.getDomainRouterIp());
            } else if (vspNic.getUseStaticIp() == Boolean.TRUE) {
                vmInterfaces.set(NuageVspAttribute.VM_INTERFACE_IPADDRESS, vspNic.getIp());
            }
            vmInterfaceList.add(vmInterfaces);
        }
        return vmInterfaceList;
    }

    @Override public void deleteVmInterface(String vmUuid, String macAddr, String vmInterfaceID) throws NuageVspException {
        try {
            api.deleteResource(NuageVspEntity.VM_INTERFACE, vmInterfaceID);
            s_logger.debug(
                    "VM Interface is getting destroyed for VM with UUID " + vmUuid + " and it exists in NuageVSP. Deleted the VM interface " + vmInterfaceID + " from Nuage VSP");
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to delete VM Interface with MAC " + macAddr + " for the VM with UUID " + vmUuid + " from NUAGE VSP. Json response from VSP REST API is  "
                    + exception.getMessage();
            s_logger.error(errorMessage, exception);
        }
    }

    @Override public void deleteVM(String vmUuid, String vmId) {
        try {
            api.deleteResource(NuageVspEntity.VM, vmId);
            s_logger.debug("VM " + vmUuid + " is getting destroyed and it exists in NuageVSP. Deleted the VM " + vmId + " from Nuage VSP");
        } catch (NuageVspApiException exception) {
            s_logger.error(
                    "VM " + vmUuid + " is getting destroyed. REST API failed to delete the VM " + vmId + " from NuageVsp. Json response from REST API is " + exception.getMessage(),
                    exception);
        }
    }

    @Override public NuageVspObject getVMDetails(String vmUuid) throws NuageVspException {
        String vmJsonString;
        try {
            vmJsonString = api.getResources(NuageVspEntity.VM, where(NuageVspAttribute.VM_UUID).eq(vmUuid));
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to execute REST API call to VSP to get VM with UUID " + vmUuid + ". So, Failed to get IP for the VM from VSP address for VM " + vmUuid
                    + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
        return Iterables.getFirst(api.parseJsonString(NuageVspEntity.VM, vmJsonString), null);
    }

    @Override public String getOrCreateVSPEnterprise(String domainUuid, String domainName, String domainPath) throws NuageVspException {
        String enterpriseId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, domainUuid);
        if (StringUtils.isBlank(enterpriseId)) {
            //Create a Enterprise corresponding to networksDomain
            enterpriseId = createEnterpriseInVSP(domainUuid, getEnterpriseName(domainName, domainPath));
        }
        return enterpriseId;
    }

    @Override public Pair<String, String> getOrCreateVSPEnterpriseAndGroup(String networksDomainName, String networksDomainPath, String networksDomainUuid,
            String networksAccountName, String networksAccountUuid, java.util.Optional<String> maybeEnterpriseId) throws NuageVspException {

        Supplier<String> findEnterpriseIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, networksDomainUuid);

        String enterpriseId = maybeEnterpriseId.orElseGet(findEnterpriseIdByUuid);

        if (StringUtils.isBlank(enterpriseId)) {
            //Create a Enterprise corresponding to networksDomain
            enterpriseId = createEnterpriseInVSP(networksDomainUuid, getEnterpriseName(networksDomainName, networksDomainPath));
        }

        //Check if user exists. If no then create an user under enterprise
        String userId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.USER, networksAccountUuid);
        if (StringUtils.isBlank(userId)) {
            userId = createUserInEnterprise(enterpriseId, networksAccountUuid, NuageVspConstants.USER_FIRST_NAME + "_" + networksAccountUuid,
                                            NuageVspConstants.USER_LAST_NAME + "_" + networksAccountUuid, NuageVspConstants.USER_EMAIL,
                                            DigestUtils.shaHex(NuageVspConstants.USER_PASSWORD));
        }

        NuageVspObject userGroup = api.getEntityByExternalId(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.GROUP, networksAccountUuid);
        if (userGroup == null) {
            //Create a Group and User that corresponds to networkAccount
            userGroup = createGroupInEnterprise(networksAccountName, networksAccountUuid, enterpriseId);
        }

        //Add the user in to group1
        api.linkRelatedEntity(userGroup, NuageVspEntity.USER, userId);

        return Pair.of(enterpriseId, userGroup.getId());
    }

    @Override public String createEnterpriseInVSP(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException {

        s_logger.debug("Enterprise with domainUuid " + enterpriseDescription + " does not exist in VSP. So, just create the new Enterprise");

        String enterpriseProfileId = createEnterpriseProfileInVsp(enterpriseExternalUuid, enterpriseDescription);

        NuageVspObject enterpriseEntity = api.createNuageVspObject(NuageVspEntity.ENTERPRISE);
        enterpriseEntity.set(NuageVspAttribute.NAME, enterpriseExternalUuid);
        enterpriseEntity.set(NuageVspAttribute.DESCRIPTION, enterpriseDescription);
        enterpriseEntity.setExternalId(enterpriseExternalUuid);
        enterpriseEntity.set(NuageVspAttribute.ENTERPRISE_PROFILE_ID, enterpriseProfileId);

        try {
            NuageVspObject enterprise = api.createResource(enterpriseEntity);
            return enterprise.getId();
        } catch (NuageVspApiException e) {
            String errorMessage = "Failed to create Enterprise in VSP using REST API. So, Enterprise could not be created in VSP " + " for domain " + enterpriseExternalUuid
                    + ".  Json response from VSP REST API is  " + e.getMessage();
            s_logger.error(errorMessage, e);
            //clean the enterprise profile id
            api.deleteQuietly(NuageVspEntity.ENTERPRISE_PROFILE, enterpriseProfileId);
            throw new NuageVspException(errorMessage);
        }
    }

    private String createEnterpriseProfileInVsp(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException {
        //Create the enterprise profile and then associate the profile to the enterprise
        String enterpriseProfileId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE_PROFILE, null, null, enterpriseExternalUuid);
        if (StringUtils.isBlank(enterpriseProfileId)) {
            NuageVspObject enterpriseProfileEntity = api.createNuageVspObject(NuageVspEntity.ENTERPRISE_PROFILE);
            enterpriseProfileEntity.set(NuageVspAttribute.NAME, enterpriseExternalUuid);
            enterpriseProfileEntity.set(NuageVspAttribute.DESCRIPTION, enterpriseDescription);
            enterpriseProfileEntity.setExternalId(enterpriseExternalUuid);

            Integer floatingIpQuota = ConfigUtil.getPropertyInteger(ConfigUtil.FLOATING_IP_QUOTA, NuageVspConstants.EnterpriseProfileDefaults.FLOATING_IP_QUOTA);
            Boolean allowGatewayMgmt = ConfigUtil.getPropertyBoolean(ConfigUtil.ALLOW_GATEWAY_MANAGEMENT, NuageVspConstants.EnterpriseProfileDefaults.ALLOW_GATEWAY_MANAGEMENT);
            Boolean allowAdvancedQOS = ConfigUtil.getPropertyBoolean(ConfigUtil.ALLOW_ADVANCED_QOS, NuageVspConstants.EnterpriseProfileDefaults.ALLOW_ADVANCED_QOS);
            enterpriseProfileEntity.set(NuageVspAttribute.ENTERPRISE_PROFILE_ADV_QOS, allowAdvancedQOS);
            enterpriseProfileEntity.set(NuageVspAttribute.ENTERPRISE_PROFILE_FLOATING_IP_QUOTA, floatingIpQuota);
            enterpriseProfileEntity.set(NuageVspAttribute.ENTERPRISE_PROFILE_GATEWAY_MGMT, allowGatewayMgmt);

            String availableFwdClass = ConfigUtil.getProperty(ConfigUtil.AVAILABLE_FWD_CLASS, NuageVspConstants.EnterpriseProfileDefaults.FWD_CLASSES);
            enterpriseProfileEntity.set(NuageVspAttribute.ENTERPRISE_PROFILE_FWD_CLASSES, Arrays.asList(availableFwdClass.split(",")));

            try {
                NuageVspObject enterpriseProfile = api.createResource(enterpriseProfileEntity);
                return enterpriseProfile.getId();
            } catch (NuageVspApiException e) {
                String errorMessage =
                        "Failed to create Enterprise Profile in VSP using REST API. So, Enterprise could not be created in VSP " + " for domain " + enterpriseExternalUuid
                                + ".  Json response from VSP REST API is  " + e.getMessage();
                s_logger.error(errorMessage, e);
                throw new NuageVspException(errorMessage);
            }
        }
        return enterpriseProfileId;
    }

    @Override public void deleteEnterpriseInVsp(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException {
        try {
            String enterpriseId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, enterpriseExternalUuid);
            if (StringUtils.isNotBlank(enterpriseId)) {
                api.deleteResource(NuageVspEntity.ENTERPRISE, enterpriseId);
                s_logger.debug("Enterprise " + enterpriseDescription + " is getting removed and it exists in NuageVSP. Deleted the enterprise " + enterpriseId + " from Nuage VSP");
                deleteEnterpriseProfileInVsp(enterpriseExternalUuid, enterpriseDescription);
            }
        } catch (NuageVspApiException e) {
            String errorMessage = "Failed to delete Enterprise in VPS using REST API. Json response from VSP REST API is " + e.getMessage();
            s_logger.error(errorMessage, e);
            throw new NuageVspException(errorMessage);
        }
    }

    private void deleteEnterpriseProfileInVsp(String enterpriseExternalUuid, String enterpriseDescription) throws NuageVspException {
        try {
            String enterpriseProfileId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE_PROFILE, null, null, enterpriseExternalUuid);
            if (StringUtils.isNotBlank(enterpriseProfileId)) {
                api.deleteResource(NuageVspEntity.ENTERPRISE_PROFILE, enterpriseProfileId);
                s_logger.debug(
                        "Enterprise Profile " + enterpriseDescription + " is getting removed and it exists in NuageVSP. " + "Deleted the enterprise profile " + enterpriseProfileId
                                + " from Nuage VSP");
            }
        } catch (NuageVspApiException e) {
            String errorMessage = "Failed to delete Enterprise Profile in VPS using REST API. Json response from VSP REST API is " + e.getMessage();
            s_logger.error(errorMessage, e);
            throw new NuageVspException(errorMessage);
        }
    }

    @Override public String createUserInEnterprise(String vsdEnterpriseId, String userNameUuid, String firstName, String lastName, String email, String password)
            throws NuageVspException {
        try {
            //create User
            NuageVspObject userEntity = api.createNuageVspObject(NuageVspEntity.USER);
            userEntity.setExternalId(userNameUuid);
            //Hack to make the user name less than 32 characters...
            userEntity.set(NuageVspAttribute.USER_USERNAME, userNameUuid.replaceAll("-", ""));
            userEntity.set(NuageVspAttribute.USER_EMAIL, email);
            userEntity.set(NuageVspAttribute.USER_PASSWORD, password);
            userEntity.set(NuageVspAttribute.USER_FIRSTNAME, firstName);
            userEntity.set(NuageVspAttribute.USER_LASTNAME, lastName);

            NuageVspObject userJson = api.createResource(NuageVspEntity.ENTERPRISE, vsdEnterpriseId, userEntity);
            s_logger.debug("Created user in VSP. Response from VSP is " + userJson);
            return userJson.getId();
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to create User for VSP enterprise " + vsdEnterpriseId + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
    }

    public void addPermission(NuageVspEntity entityType, String entityId, Set<String> groupIds) throws NuageVspException {
        try {
            //Add users to the group
            api.setRelatedEntities(entityType, entityId, NuageVspEntity.GROUP, groupIds);
            s_logger.debug("Added permission for entity " + entityType + " id " + entityId + " with groups " + groupIds);
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to add permission for entity " + entityType + " id " + entityId + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
    }

    @Override public NuageVspObject createGroupInEnterprise(String projectOrAccountName, String projectOrAccountUuid, String vsdEnterpriseId) throws NuageVspException {
        try {
            //create group
            NuageVspObject groupEntity = api.createNuageVspObject(NuageVspEntity.GROUP);
            groupEntity.set(NuageVspAttribute.NAME, projectOrAccountUuid);
            groupEntity.set(NuageVspAttribute.DESCRIPTION, projectOrAccountName);
            groupEntity.setExternalId(projectOrAccountUuid);

            NuageVspObject groupJson = api.createResource(NuageVspEntity.ENTERPRISE, vsdEnterpriseId, groupEntity);
            s_logger.debug("Created group for project or account " + projectOrAccountUuid + " in VSP . Response from VSP is " + groupJson);
            return groupJson;
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to create Group for project or account " + projectOrAccountUuid + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
    }

    public void addUserToGroup(String vsdGroupId, String userId) throws NuageVspException {
        try {
            //Add users to the group
            api.linkRelatedEntity(NuageVspEntity.GROUP, vsdGroupId, NuageVspEntity.USER, userId);
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to add Users for project or account in VSD " + vsdGroupId + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
    }

    @Override public NetworkRelatedVsdIds createNetworkConfigurationWithDefaultACLs(VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) throws NuageVspException {
        Pair<String, String> enterpriseAndGroupId = getOrCreateVSPEnterpriseAndGroup(vspNetwork);

        s_logger.debug("Create or find a VPC/Isolated/Shared network associated to network " + vspNetwork.getName() + " in VSP");
        boolean isVpc = vspNetwork.isVpc();
        Pair<String, String> vpcOrSubnetInfo = vspNetwork.getVpcOrSubnetInfo();

        String domainTemplateId = null;

        Pair<String, String> subnetIdAndZoneId;
        String vsdDomainTemplateId = null;
        String vsdDomainTemplateName = null;
        Supplier<String> findDomainIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), NuageVspEntity.DOMAIN,
                                                                               vpcOrSubnetInfo.getLeft());

        if (StringUtils.isNotBlank(vspNetwork.getDomainTemplateName())) {
            String vsdDomainTemplateEntity = findEntityUsingFilter(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), NuageVspEntity.DOMAIN_TEMPLATE,
                                                                   NuageVspAttribute.NAME, vspNetwork.getDomainTemplateName());
            NuageVspObject vsdDomainTemplate = api.getFirstJsonEntity(NuageVspEntity.DOMAIN_TEMPLATE, vsdDomainTemplateEntity);

            if (vsdDomainTemplate != null) {
                vsdDomainTemplateId = vsdDomainTemplate.getId();
                vsdDomainTemplateName = vsdDomainTemplate.getName();
            }

            if (StringUtils.isBlank(vsdDomainTemplateId) || !StringUtils.equals(vsdDomainTemplateName, vspNetwork.getDomainTemplateName())) {
                String errorMessage = "Preconfigured DomainTemplate '" + vspNetwork.getDomainTemplateName() + "' could not be found.";
                if (isVpc) {
                    errorMessage += " Please remove the VPC Tier before trying again.";
                }
                s_logger.error(errorMessage);
                throw new NuageVspException(errorMessage);
            }
        }

        boolean predefinedDomainTemplateSet = StringUtils.isNotBlank(vsdDomainTemplateId);
        boolean createDefaultAcls = !predefinedDomainTemplateSet; // don't set defaults if domain template set

        try {
            if (predefinedDomainTemplateSet) {
                String domainId = vspNetwork.getNetworkRelatedVsdIds()
                                            .getVsdDomainId()
                                            .orElseGet(findDomainIdByUuid);

                if (StringUtils.isNotBlank(domainId)) {
                    subnetIdAndZoneId = validateDomain(domainId, vspNetwork, vspDhcpOptions);

                    return new NetworkRelatedVsdIds.Builder().vsdDomainId(domainId)
                                                             .vsdEnterpriseId(enterpriseAndGroupId.getLeft())
                                                             .vsdSubnetId(subnetIdAndZoneId.getLeft())
                                                             .vsdZoneId(subnetIdAndZoneId.getRight())
                                                             .build();
                } else {
                    return createDomainZoneAndSubnet(enterpriseAndGroupId, vsdDomainTemplateId, createDefaultAcls, vspNetwork, vspDhcpOptions);
                }
            } else {
                domainTemplateId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), NuageVspEntity.DOMAIN_TEMPLATE, vpcOrSubnetInfo.getLeft());
                if (StringUtils.isNotBlank(domainTemplateId)) {
                    s_logger.debug("Domain Template " + domainTemplateId + " already exists for network " + vspNetwork.getName() + " in VSP");

                    String domainId = vspNetwork.getNetworkRelatedVsdIds()
                                                .getVsdDomainId()
                                                .orElseGet(findDomainIdByUuid);

                    if (StringUtils.isNotBlank(domainId)) {
                        subnetIdAndZoneId = validateDomain(domainId, vspNetwork, vspDhcpOptions);
                    } else {
                        throw new NuageVspException("Domain is not found under the DomainTemplate " + domainTemplateId + " for network " + vspNetwork.getName()
                                                            + " in VSP. There is a network sync issue with VSD");
                    }

                    return new NetworkRelatedVsdIds.Builder().vsdDomainId(domainId)
                                                             .vsdEnterpriseId(enterpriseAndGroupId.getLeft())
                                                             .vsdSubnetId(subnetIdAndZoneId.getLeft())
                                                             .vsdZoneId(subnetIdAndZoneId.getRight())
                                                             .build();
                } else {

                    NuageVspObject domainTemplateEntity = api.createNuageVspObject(NuageVspEntity.DOMAIN_TEMPLATE);
                    domainTemplateEntity.set(NuageVspAttribute.NAME, vpcOrSubnetInfo.getLeft());
                    domainTemplateEntity.set(NuageVspAttribute.DESCRIPTION, vpcOrSubnetInfo.getRight());
                    domainTemplateEntity.setExternalId(vpcOrSubnetInfo.getLeft());

                    try {
                        NuageVspObject domainTemplateJson = api.createResource(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), domainTemplateEntity);
                        s_logger.debug("Created DomainTemplate for network " + vspNetwork.getName() + " in VSP . Response from VSP is " + domainTemplateJson);
                        domainTemplateId = domainTemplateJson.getId();
                    } catch (NuageVspApiException exception) {
                        throw new NuageVspException(
                                " Failed to create DomainTemplate for network " + vspNetwork.getName() + ". Json response from VSP REST API is " + exception.getMessage());
                    }

                    return createDomainZoneAndSubnet(enterpriseAndGroupId, domainTemplateId, createDefaultAcls, vspNetwork, vspDhcpOptions);
                }
            }
        } catch (NuageVspException e) {
            api.deleteQuietly(NuageVspEntity.DOMAIN_TEMPLATE, domainTemplateId);
            throw e;
        }

    }

    private Pair<String, String> getOrCreateVSPEnterpriseAndGroup(VspNetwork vspNetwork) {
        final VspDomain vspDomain = vspNetwork.getVspDomain();
        return getOrCreateVSPEnterpriseAndGroup(vspDomain.getName(),
                                                vspDomain.getPath(),
                                                vspDomain.getUuid(),
                                                vspNetwork.getAccountName(),
                                                vspNetwork.getAccountUuid(),
                                                vspNetwork.getNetworkRelatedVsdIds().getVsdEnterpriseId());
    }

    private Pair<String, String> validateDomain(String domainId, VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) throws NuageVspException {
        Supplier<String> findSubnetIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.DOMAIN, domainId, NuageVspEntity.SUBNET, vspNetwork.getSubnetExternalId());

        String subnetId = vspNetwork.getNetworkRelatedVsdIds()
                                    .getVsdSubnetId()
                                    .orElseGet(findSubnetIdByUuid);
        String zoneId = null;

        if (!vspNetwork.shouldReuseDomain() && StringUtils.isBlank(subnetId)) {
            throw new NuageVspException(
                    "Subnet is not found under the Domain " + domainId + " for network " + vspNetwork.getName() + " in VSP. There is a network sync issue with VSD");
        } else if (vspNetwork.shouldReuseDomain()) {
            String zoneExternalUuid = vspNetwork.getVpcOrSubnetInfo()
                                                .getLeft();

            Supplier<String> findZoneIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.DOMAIN, domainId, NuageVspEntity.ZONE, zoneExternalUuid);

            zoneId = vspNetwork.getNetworkRelatedVsdIds()
                               .getVsdSubnetId()
                               .orElseGet(findZoneIdByUuid);

            if (StringUtils.isBlank(zoneId)) {
                throw new NuageVspException(
                        "Zone corresponding to network " + zoneExternalUuid + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
            } else if (StringUtils.isBlank(subnetId)) {
                subnetId = createL3Subnet(zoneId, vspNetwork, vspDhcpOptions);
            } else {
                s_logger.debug(" Subnet " + subnetId + " already exists for network " + vspNetwork.getName() + " in VSP");
            }
        } else {
            s_logger.debug(" Domain and Subnet " + subnetId + " already exists for network " + vspNetwork.getName() + " in VSP");
        }

        return Pair.of(subnetId, zoneId);
    }

    private NetworkRelatedVsdIds createDomainZoneAndSubnet(Pair<String, String> enterpriseAndGroupId, String domainTemplateId, boolean createDefaultAcls, VspNetwork vspNetwork,
            VspDhcpDomainOption vspDhcpOptions) throws NuageVspException {
        String domainId;
        String ingressAclTemplId = null;
        String egressAclTemplId = null;
        Pair<String, String> vpcOrSubnetInfo = vspNetwork.getVpcOrSubnetInfo();
        try {
            //Now instantiate the domain template
            NuageVspObject domainEntity = api.createNuageVspObject(NuageVspEntity.DOMAIN);
            domainEntity.set(NuageVspAttribute.NAME, vpcOrSubnetInfo.getLeft());
            domainEntity.set(NuageVspAttribute.DESCRIPTION, vpcOrSubnetInfo.getRight());
            domainEntity.setExternalId(vpcOrSubnetInfo.getLeft());
            domainEntity.set(NuageVspAttribute.TEMPLATE_ID, domainTemplateId);
            domainEntity.set(NuageVspAttribute.DOMAIN_PATENABLED, "ENABLED");
            domainEntity.set(NuageVspAttribute.DOMAIN_UNDERLAYENABLED, "ENABLED");

            NuageVspObject domainJson = api.createResource(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), domainEntity);
            s_logger.debug("Created Domain for network " + vspNetwork.getName() + " in VSP . Response from VSP is " + domainJson);
            domainId = domainJson.getId();
        } catch (NuageVspException exception) {
            throw new NuageVspException(
                    "Failed to instantiate DomainTemplate for network " + vspNetwork.getName() + ". Json response from VSP REST API is " + exception.getMessage());
        }

        //Create default ingress and egress ACLs
        if (createDefaultAcls) {
            aclClient.createDefaultRules(vspNetwork, NuageVspEntity.DOMAIN, domainId);
        }

        String zoneId;
        try {
            //Now create the Zone under the domain
            NuageVspObject zoneEntity = api.createNuageVspObject(NuageVspEntity.ZONE);
            zoneEntity.set(NuageVspAttribute.NAME, NuageVspConstants.ZONE_NAME + "_" + vpcOrSubnetInfo.getLeft());
            zoneEntity.set(NuageVspAttribute.DESCRIPTION, vpcOrSubnetInfo.getRight());
            zoneEntity.setExternalId(vpcOrSubnetInfo.getLeft());
            zoneEntity.set(NuageVspAttribute.ZONE_PUBLIC, vspNetwork.isShared());

            NuageVspObject zoneJson = api.createResource(NuageVspEntity.DOMAIN, domainId, zoneEntity);
            zoneId = zoneJson.getId();
            s_logger.debug("Created Zone for network " + vspNetwork.getName() + " in VSP. Response from VSP is " + zoneJson);

            //set permission to use the Zone
            Set<String> groupIds = ImmutableSet.of(enterpriseAndGroupId.getRight());
            addPermission(NuageVspEntity.ZONE, zoneId, groupIds);
        } catch (NuageVspException exception) {
            throw new NuageVspException("Failed to create Zone for network " + vspNetwork.getName() + ". Json response from VSP REST API is " + exception.getMessage());
        }

        String subnetId = createL3Subnet(zoneId, vspNetwork, vspDhcpOptions);

        return new NetworkRelatedVsdIds.Builder().vsdEnterpriseId(enterpriseAndGroupId.getLeft())
                                                 .vsdZoneId(zoneId)
                                                 .vsdSubnetId(subnetId)
                                                 .vsdDomainId(domainId)
                                                 .build();
    }

    private String createL3Subnet(String zoneId, VspNetwork vspNetwork, VspDhcpDomainOption vspDhcpOptions) throws NuageVspException {
        String subnetId;
        try {
            SubnetUtils.SubnetInfo subnetInfo = new SubnetUtils(vspNetwork.getCidr()).getInfo();

            //Now create the Subnet under the Zone
            NuageVspObject subnetEntity = api.createNuageVspObject(NuageVspEntity.SUBNET);
            subnetEntity.set(NuageVspAttribute.NAME, vspNetwork.getSubnetExternalId());
            subnetEntity.set(NuageVspAttribute.DESCRIPTION, vspNetwork.getName());
            subnetEntity.set(NuageVspAttribute.SUBNET_ADDRESS, subnetInfo.getNetworkAddress());
            subnetEntity.set(NuageVspAttribute.SUBNET_NETMASK, subnetInfo.getNetmask());
            subnetEntity.set(NuageVspAttribute.SUBNET_GATEWAY, vspNetwork.getGateway());
            subnetEntity.setExternalId(vspNetwork.getSubnetExternalId());

            DhcpOptions dhcpOptions = null;
            if (vspDhcpOptions != null) {
                if (vspDhcpOptions.getVrIsDnsProvider()) {
                    dhcpOptions = new DhcpOptions(vspNetwork.getVirtualRouterIp(), vspDhcpOptions.getDnsServers(), vspDhcpOptions.getNetworkDomain());
                } else {
                    dhcpOptions = new DhcpOptions(null, vspDhcpOptions.getDnsServers(), null);
                }
            }

            if (vspNetwork.isShared()) {
                String sharedL3SubnetId = createL3SubnetSharedResources(vspNetwork, subnetInfo, dhcpOptions);
                subnetEntity.set(NuageVspAttribute.ASSOC_SHARED_NTWK_ID, sharedL3SubnetId);
            }

            NuageVspObject subnetJson = api.createResource(NuageVspEntity.ZONE, zoneId, subnetEntity);
            s_logger.debug("Created subnet for network " + vspNetwork.getName() + " in VSP. Response from VSP is " + subnetJson);

            subnetId = subnetJson.getId();
            if (!vspNetwork.isShared() && dhcpOptions != null) {
                createDhcpOptions(true, NuageVspEntity.SUBNET, subnetId, vspNetwork.getUuid(), vspNetwork, dhcpOptions);
            }
        } catch (NuageVspException exception) {
            throw new NuageVspException("Failed to create Subnet for network " + vspNetwork.getName() + ". Json response from VSP REST API is " + exception.getMessage());
        }
        return subnetId;
    }

    private String createL3SubnetSharedResources(VspNetwork vspNetwork, SubnetUtils.SubnetInfo subnetInfo, DhcpOptions dhcpOptions) throws NuageVspException {
        String sharedL3SubnetId = null;
        Set<SharedNetworkType> resourceTypes = vspNetwork.isPublicAccess() ?
                EnumSet.of(SharedNetworkType.FLOATING, SharedNetworkType.PUBLIC) :
                EnumSet.of(SharedNetworkType.PUBLIC);

        String externalIdFromGateway = vspNetwork.getSubnetExternalId();
        for (SharedNetworkType resourceType : resourceTypes) {
            String sharedResourceJson = findSharedResource(true, resourceType, externalIdFromGateway, vspNetwork.getGateway(), subnetInfo.getNetmask());
            if (StringUtils.isNotBlank(sharedResourceJson)) {
                s_logger.debug("Shared resource with UUID " + externalIdFromGateway + " and gateway " + vspNetwork.getGateway() + " is already created");

                if (resourceType == SharedNetworkType.PUBLIC) {
                    NuageVspObject sharedResource = Iterables.getFirst(api.parseJsonString(NuageVspEntity.SHARED_NETWORK, sharedResourceJson), null);
                    sharedL3SubnetId = sharedResource.get(NuageVspAttribute.ID);
                }
                continue;
            }

            for (VspAddressRange vspAddressRange : vspNetwork.getAddressRanges()) {
                String externalId = UuidUtils.generateUuidFromExternalIdAndIp(vspNetwork.getUuid(), vspAddressRange.getGateway());
                sharedResourceJson = findSharedResource(true, resourceType, externalId, vspAddressRange.getGateway(), vspAddressRange.getNetmask());
                if (StringUtils.isNotBlank(sharedResourceJson)) {
                    break;
                }
            }

            NuageVspObject sharedResource = Iterables.getFirst(api.parseJsonString(NuageVspEntity.SHARED_NETWORK, sharedResourceJson), null);
            String sharedResourceParentId = sharedResource != null ? (String)sharedResource.get(NuageVspAttribute.PARENT_ID) : null;
            String sharedResourceId = createSharedResource(resourceType, sharedResourceParentId, externalIdFromGateway, vspNetwork.getGateway(), subnetInfo.getNetmask(),
                                                           (resourceType.equals(SharedNetworkType.FLOATING)?vspNetwork.isVlanUnderlay():false), vspNetwork.getName());

            if (dhcpOptions != null) {
                createDhcpOptions(true, NuageVspEntity.SHARED_NETWORK, sharedResourceId, externalIdFromGateway, vspNetwork, dhcpOptions);
            }

            if (resourceType == SharedNetworkType.PUBLIC) {
                sharedL3SubnetId = sharedResourceId;
            }
        }
        return sharedL3SubnetId;
    }

    @Override public void createDhcpOptions(ExistingDhcpOptionStrategy existingOptionStrategy, NetworkDetails attachedNetworkDetails, VspNetwork vspNetwork,
            DhcpOptions dhcpOptions) throws NuageVspException {
        if (vspNetwork.isShared()) {
            String externalIdFromGateway = vspNetwork.getSubnetExternalId();
            String sharedResourceJson = findSharedResource(true, SharedNetworkType.PUBLIC, externalIdFromGateway, vspNetwork.getGateway(),
                                                           NetUtils.getCidrNetmask(vspNetwork.getCidr()));
            String sharedResourceId = getEntityId(sharedResourceJson, NuageVspEntity.SHARED_NETWORK);
            createDhcpOptions(existingOptionStrategy, NuageVspEntity.SHARED_NETWORK, externalIdFromGateway, sharedResourceId, vspNetwork, dhcpOptions);
        } else {
            createDhcpOptions(existingOptionStrategy, attachedNetworkDetails.getSubnetType(), vspNetwork.getUuid(), attachedNetworkDetails.getSubnetId(), vspNetwork, dhcpOptions);
        }
    }

    public void createDhcpOptions(boolean isCreateDhcpOption, NuageVspEntity nuageVspEntity, String nuageVspEntityId, String nuageVspEntityUuid, VspNetwork vspNetwork,
            DhcpOptions dhcpOptions) throws NuageVspException {

        ExistingDhcpOptionStrategy existingOptionStrategy = isCreateDhcpOption ? ExistingDhcpOptionStrategy.IGNORE : ExistingDhcpOptionStrategy.FETCH_AND_DELETE;
        createDhcpOptions(existingOptionStrategy, nuageVspEntity, nuageVspEntityUuid, nuageVspEntityId, vspNetwork, dhcpOptions);
    }

    @Override public void createDhcpOptions(ExistingDhcpOptionStrategy existingOptionStrategy, NuageVspEntity nuageVspEntity, String nuageVspEntityUuid, String nuageVspEntityId,
            VspNetwork vspNetwork, DhcpOptions dhcpOptions) throws NuageVspException {

        List<NuageVspObject> existingDhcpOptions = Arrays.asList();
        if (existingOptionStrategy.isFetch()) {
            //get the DHCP option information and check if exists. If yes, then see if the DHCP option is modified
            String dhcpOptionsJson = findEntityByExternalUuid(nuageVspEntity, nuageVspEntityId, NuageVspEntity.DHCP_OPTIONS, nuageVspEntityUuid);
            if (!StringUtils.isBlank(dhcpOptionsJson)) {
                existingDhcpOptions = api.parseJsonString(NuageVspEntity.DHCP_OPTIONS, dhcpOptionsJson);
            }
        }

        for (final DhcpOption dhcpOption : dhcpOptions.getOptions()) {
            //get the DHCP option information and check if exists. If yes, then see if the DHCP option is modified
            java.util.Optional<NuageVspObject> optionalExistingDhcpOption = existingDhcpOptions.stream()
                                                                                               .filter(option -> (Integer)option.get(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_TYPE)
                                                                                                       == dhcpOption.getCode())
                                                                                               .findFirst();

            if (optionalExistingDhcpOption.isPresent()) {
                // Update exiting DhcpOption if current value is different
                NuageVspObject existingDhcpOption = optionalExistingDhcpOption.get();
                boolean dhcpOptionValueChanged;

                if (dhcpOption.getOptionType() == Dhcp.DhcpOptionType.RAW) {
                    String dhcpOptionValue = existingDhcpOption.get(NuageVspAttribute.DHCP_OPTIONS_VALUE);
                    dhcpOptionValueChanged = !dhcpOptionValue.equals(dhcpOption.getValueAsHex());
                } else {
                    List<String> dhcpOptionValue = existingDhcpOption.get(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_VALUES);
                    dhcpOptionValueChanged = !dhcpOptionValue.equals(dhcpOption.getValue());
                }

                if (dhcpOptionValueChanged) {
                    updateDhcpOption(existingDhcpOption, dhcpOption);
                    api.updateResource(existingDhcpOption);
                    s_logger.debug("Network (" + vspNetwork.getName() + ") DNS server's setting is changed to new value " + dhcpOption.getValue()
                                           + ". So, the DHCPOption for the network is updated");
                }

                existingDhcpOptions.remove(existingDhcpOption);
            } else {
                // Create a new DhcpOption
                NuageVspObject dhcpOptionToCreate = createDhcpOption(nuageVspEntityUuid, dhcpOption);
                NuageVspObject dhcpResponseJson = api.createResource(nuageVspEntity, nuageVspEntityId, dhcpOptionToCreate);
                s_logger.debug("Created DHCP options for network " + vspNetwork.getName() + " in VSP. Response from VSP is " + dhcpResponseJson);
            }
        }

        if (existingDhcpOptions != null && existingOptionStrategy.isDeleteUnspecified()) {
            // Delete all non updated dhcp options
            for (NuageVspObject dhcpOptionToRemove : existingDhcpOptions) {
                s_logger.debug("Network (" + nuageVspEntityUuid + ") DNS server's setting " + dhcpOptionToRemove.get(NuageVspAttribute.DHCP_OPTIONS_VALUE)
                                       + " is removed. So, delete the DHCPOption for the network");
                String dhcpOptionId = dhcpOptionToRemove.getId();
                api.deleteQuietly(NuageVspEntity.DHCP_OPTIONS, dhcpOptionId);
            }
        }
    }

    @Override public void removeAllDhcpOptionsWithCode(NuageVspEntity nuageVspEntity, String externalUuid, String entityId, Set<Integer> dhcpOptionCodesToRemove)
            throws NuageVspException {
        List<NuageVspObject> foundDhcpOptions = api.parseJsonString(NuageVspEntity.DHCP_OPTIONS,
                                                                    findEntityByExternalUuid(nuageVspEntity, entityId, NuageVspEntity.DHCP_OPTIONS, externalUuid));

        foundDhcpOptions.stream()
                        .filter(vspObject -> dhcpOptionCodesToRemove.contains(vspObject.<Integer>get(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_TYPE)))
                        .forEach(api::deleteQuietly);
    }

    private NuageVspObject updateDhcpOption(NuageVspObject existingDhcpOption, DhcpOption dhcpOption) {
        //Depending on the type use a different API
        if (dhcpOption.getOptionType() == Dhcp.DhcpOptionType.RAW) {
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_LENGTH, dhcpOption.getLengthAsHex());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_TYPE, dhcpOption.getCodeAsHex());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_VALUE, dhcpOption.getValueAsHex());
            existingDhcpOption.unset(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_TYPE);
            existingDhcpOption.unset(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_VALUES);
        } else {
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_TYPE, dhcpOption.getCode());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_VALUES, dhcpOption.getValue());
            existingDhcpOption.unset(NuageVspAttribute.DHCP_OPTIONS_LENGTH);
            existingDhcpOption.unset(NuageVspAttribute.DHCP_OPTIONS_TYPE);
            existingDhcpOption.unset(NuageVspAttribute.DHCP_OPTIONS_VALUE);
        }
        return existingDhcpOption;
    }

    private NuageVspObject createDhcpOption(String externalId, DhcpOption dhcpOption) {
        NuageVspObject existingDhcpOption = api.createNuageVspObject(NuageVspEntity.DHCP_OPTIONS);
        //Depending on the type use a different API
        if (dhcpOption.getOptionType() == Dhcp.DhcpOptionType.RAW) {
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_LENGTH, dhcpOption.getLengthAsHex());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_TYPE, dhcpOption.getCodeAsHex());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_VALUE, dhcpOption.getValueAsHex());
        } else {
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_TYPE, dhcpOption.getCode());
            existingDhcpOption.set(NuageVspAttribute.DHCP_OPTIONS_ACTUAL_VALUES, dhcpOption.getValue());
        }
        existingDhcpOption.setExternalId(externalId);
        return existingDhcpOption;
    }

    private String getPaddedHexValue(String dnsServer) {
        String value = Long.toHexString(ip2Long(dnsServer));
        int valueLength = 8 - value.length();
        if (valueLength > 0) {
            StringBuilder pad = new StringBuilder();
            for (int i = 0; i < valueLength; i++) {
                pad.append("0");
            }
            value = pad + value;
        }
        return value;
    }

    private long ip2Long(String ip) {
        String[] tokens = ip.split("[.]");
        assert (tokens.length == 4);
        long result = 0;
        for (String token : tokens) {
            try {
                result = (result << 8) | Integer.parseInt(token);
            } catch (NumberFormatException e) {
                throw new RuntimeException("Incorrect number", e);
            }
        }

        return result;
    }

    @Override public VspNetwork findIsolatedL3Network(VspNetwork vspNetwork) {
        String vsdSubnetId = vspNetwork.getNetworkRelatedVsdIds().getVsdSubnetId().orElseThrow(() -> new IllegalArgumentException("vsdSubnetId missing"));

        final NuageVspObject subnet = api.getResource(NuageVspEntity.SUBNET, vsdSubnetId);
        if (subnet == null) {
            throw new NuageVspException("Unable to find Subnet by id");
        }
        String vsdZoneId = subnet.get(NuageVspAttribute.PARENT_ID);

        final NuageVspObject zone = api.getResource(NuageVspEntity.ZONE, vsdZoneId);
        String vsdDomainId = zone.get(NuageVspAttribute.PARENT_ID);

        final NuageVspObject domain = api.getResource(NuageVspEntity.ZONE, vsdZoneId);
        final String vsdEnterpriseId = domain.get(NuageVspAttribute.PARENT_ID);

        final Boolean hasCorrectEnterpriseId = vspNetwork.getNetworkRelatedVsdIds()
                                           .getVsdEnterpriseId()
                                           .map(s -> s.equals(vsdEnterpriseId))
                                           .orElse(true);

        if (!hasCorrectEnterpriseId) {
            throw new NuageVspException("Specified Subnet id belongs to an other enterprise.");
        }

        final String cidr = new SubnetUtils(subnet.get(NuageVspAttribute.SUBNET_ADDRESS),
                                            subnet.get(NuageVspAttribute.SUBNET_NETMASK)).getInfo()
                                                                                                  .getCidrSignature();

        NetworkRelatedVsdIds ids = new NetworkRelatedVsdIds.Builder()
                .vsdSubnetId(vsdSubnetId)
                .vsdZoneId(vsdZoneId)
                .vsdDomainId(vsdDomainId)
                .vsdEnterpriseId(vsdEnterpriseId)
                .build();

        return new VspNetwork.Builder()
                .fromObject(vspNetwork)
                .networkType(VspNetwork.NetworkType.L3)
                .cidr(cidr)
                .gateway(subnet.get(NuageVspAttribute.SUBNET_GATEWAY))
                .name(subnet.getName())
                .networkRelatedVsdIds(ids)
                .build();

    }


    @Override public NetworkRelatedVsdIds createIsolatedL2NetworkWithDefaultACLs(VspNetwork vspNetwork) throws NuageVspException {
        Pair<String, String> enterpriseAndGroupId = getOrCreateVSPEnterpriseAndGroup(vspNetwork);
        Set<String> groupIds = ImmutableSet.of(enterpriseAndGroupId.getRight());
        SubnetUtils.SubnetInfo subnetInfo = new SubnetUtils(vspNetwork.getCidr()).getInfo();

        s_logger.debug("Create or find Isolated L2 Domain for network " + vspNetwork.getUuid() + " in VSP");

        NetworkRelatedVsdIds networkRelatedVsdIds = vspNetwork.getNetworkRelatedVsdIds();
        String l2DomainId = null;
        StringBuffer errorMessage = new StringBuffer();

        String l2DomainTemplateId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), NuageVspEntity.L2DOMAIN_TEMPLATE, vspNetwork.getUuid());
        if (StringUtils.isNotBlank(l2DomainTemplateId)) {
            s_logger.debug("L2Domain Template " + l2DomainTemplateId + " already exists.");

            Supplier<String> findL2DomainByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), NuageVspEntity.L2DOMAIN,
                                                                                   vspNetwork.getUuid());
            l2DomainId = networkRelatedVsdIds.getVsdDomainId()
                                             .orElseGet(findL2DomainByUuid);

            if (StringUtils.isNotBlank(l2DomainId)) {
                s_logger.debug("L2Domain " + l2DomainId + " already exists for network " + vspNetwork.getName() + " in VSP.");
            } else {
                errorMessage.append("L2Domain is not found under the L2DomainTemplate ")
                            .append(l2DomainTemplateId)
                            .append(" for network ")
                            .append(vspNetwork.getName())
                            .append(" in VSP. There is a network sync issue with VSD");
            }
        } else {

            NuageVspObject l2DomainTemplateEntity = api.createNuageVspObject(NuageVspEntity.L2DOMAIN_TEMPLATE);
            l2DomainTemplateEntity.set(NuageVspAttribute.NAME, vspNetwork.getUuid());
            l2DomainTemplateEntity.set(NuageVspAttribute.DESCRIPTION, vspNetwork.getName());
            l2DomainTemplateEntity.setExternalId(vspNetwork.getUuid());
            l2DomainTemplateEntity.set(NuageVspAttribute.L2DOMAIN_TEMPLATE_ADDRESS, subnetInfo.getNetworkAddress());
            l2DomainTemplateEntity.set(NuageVspAttribute.L2DOMAIN_TEMPLATE_NETMASK, subnetInfo.getNetmask());
            l2DomainTemplateEntity.set(NuageVspAttribute.L2DOMAIN_TEMPLATE_GATEWAY, vspNetwork.getGateway());

            try {
                NuageVspObject l2DomainTemplateJson = api.createResource(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), l2DomainTemplateEntity);
                s_logger.debug("Created L2DomainTemplate for network " + vspNetwork.getName() + " in VSP . Response from VSP is " + l2DomainTemplateJson);
                l2DomainTemplateId = l2DomainTemplateJson.getId();
            } catch (NuageVspApiException exception) {
                String error = "Failed to create L2DomainTemplate for network " + vspNetwork.getName() + ".  Json response from VSP REST API is  " + exception.getMessage();
                s_logger.error(error, exception);
                throw new NuageVspException(error);
            }

            try {
                if (errorMessage.length() == 0) {
                    //Now instantiate the L2domain template
                    NuageVspObject l2DomainEntity = api.createNuageVspObject(NuageVspEntity.L2DOMAIN);
                    l2DomainEntity.set(NuageVspAttribute.NAME, vspNetwork.getUuid());
                    l2DomainEntity.set(NuageVspAttribute.DESCRIPTION, vspNetwork.getName());
                    l2DomainEntity.setExternalId(vspNetwork.getUuid());
                    l2DomainEntity.set(NuageVspAttribute.TEMPLATE_ID, l2DomainTemplateId);

                    NuageVspObject l2DomainJson = api.createResource(NuageVspEntity.ENTERPRISE, enterpriseAndGroupId.getLeft(), l2DomainEntity);
                    s_logger.debug("Created L2Domain for network " + vspNetwork.getName() + " in VSP . Response from VSP is " + l2DomainJson);
                    l2DomainId = l2DomainJson.getId();

                    l2DomainEntity.setId(l2DomainId);

                    //set permission to use the Subnet
                    try {
                        addPermission(NuageVspEntity.L2DOMAIN, l2DomainId, groupIds);
                    } catch (Exception e) {
                        errorMessage.append(e.getMessage());
                    }
                }
            } catch (NuageVspApiException exception) {
                errorMessage.append("Failed to instantiate L2DomainTemplate for network ")
                            .append(vspNetwork.getName())
                            .append(".  Json response from VSP REST API is  ")
                            .append(exception.getMessage());
            }

            //Create default ingress and egress ACLs
            aclClient.createDefaultRules(vspNetwork, NuageVspEntity.L2DOMAIN_TEMPLATE, l2DomainTemplateId);
        }

        if (errorMessage.length() != 0) {
            s_logger.error(errorMessage);
            api.deleteQuietly(NuageVspEntity.L2DOMAIN_TEMPLATE, l2DomainTemplateId);
            throw new NuageVspException(errorMessage.toString());
        }

        return new NetworkRelatedVsdIds.Builder().vsdEnterpriseId(enterpriseAndGroupId.getLeft())
                                                 .vsdDomainId(l2DomainId)
                                                 .build();
    }

    @Override public String getEnterprise(String domainUuid) throws NuageVspException {
        String enterpriseId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, domainUuid);
        if (StringUtils.isBlank(enterpriseId)) {
            String errorMessage =
                    "Enterprise corresponding to CS domain " + domainUuid + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network";
            s_logger.error(errorMessage);
            throw new NuageVspException(errorMessage);
        }
        return enterpriseId;
    }

    @Override public Pair<String, String> getIsolatedSubNetwork(String enterpriseId, VspNetwork vspNetwork) throws NuageVspException {
        return getIsolatedSubNetwork(true, enterpriseId, vspNetwork);
    }

    @Override public Pair<String, String> getIsolatedSubNetwork(boolean throwExceptionWhenNotFound, String enterpriseId, VspNetwork vspNetwork) throws NuageVspException {
        Pair<String, String> vpcOrSubnetInfo = vspNetwork.getVpcOrSubnetInfo();

        //Check if L3 DomainTemplate exists
        try {
            Supplier<String> findDomainIdByUuid = () -> getIsolatedDomain(enterpriseId, NuageVspEntity.DOMAIN, vpcOrSubnetInfo.getLeft());

            String domainId = vspNetwork.getNetworkRelatedVsdIds()
                                        .getVsdDomainId()
                                        .orElseGet(findDomainIdByUuid);
            if (domainId == null) {
                return null;
            }

            Supplier<String> findZoneIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.DOMAIN, domainId, NuageVspEntity.ZONE, vpcOrSubnetInfo.getLeft());

            String zoneId = vspNetwork.getNetworkRelatedVsdIds()
                                      .getVsdZoneId()
                                      .orElseGet(findZoneIdByUuid);

            if (StringUtils.isBlank(zoneId)) {
                if (throwExceptionWhenNotFound) {
                    throw new NuageVspException("Zone corresponding to network " + vpcOrSubnetInfo.getLeft()
                                                        + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
                }
                return null;
            }

            Supplier<String> findSubnetIdByUuid = () -> findEntityIdByExternalUuid(NuageVspEntity.ZONE, zoneId, NuageVspEntity.SUBNET, vspNetwork.getSubnetExternalId());

            String subnetId = vspNetwork.getNetworkRelatedVsdIds()
                                        .getVsdSubnetId()
                                        .orElseGet(findSubnetIdByUuid);

            if (StringUtils.isBlank(subnetId)) {
                if (throwExceptionWhenNotFound) {
                    throw new NuageVspException("Subnet corresponding to network " + vspNetwork.getUuid()
                                                        + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
                }
                return null;
            }
            return Pair.of(domainId, subnetId);
        } catch (NuageVspApiException exception) {
            String errorMessage = "Failed to get Subnet corresponding to network " + vspNetwork.getUuid() + ".  Json response from VSP REST API is  " + exception.getMessage();
            s_logger.error(errorMessage, exception);
            throw new NuageVspException(errorMessage);
        }
    }

    @Override public String getIsolatedDomain(String enterpriseId, NuageVspEntity attachedNetworkType, String vpcOrSubnetUuid) throws NuageVspException {
        return getIsolatedDomain(true, enterpriseId, attachedNetworkType, vpcOrSubnetUuid);
    }

    @Override public String getIsolatedDomain(boolean throwExceptionWhenNotFound, String enterpriseId, NuageVspEntity attachedNetworkType, String vpcOrSubnetUuid)
            throws NuageVspException {
        String domainId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, attachedNetworkType, vpcOrSubnetUuid);
        if (StringUtils.isBlank(domainId)) {
            if (throwExceptionWhenNotFound) {
                throw new NuageVspException(attachedNetworkType + " corresponding to network " + vpcOrSubnetUuid
                                                    + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
            }
            return null;
        }
        return domainId;
    }

    public List<NuageVspObject> getACLAssociatedToDomain(boolean ingress, NuageVspEntity domainType, String domainUuid, String networkUuid) throws NuageVspException {
        NuageVspEntity aclType = ingress ? NuageVspEntity.INGRESS_ACLTEMPLATES : NuageVspEntity.EGRESS_ACLTEMPLATES;
        String aclTemplates = api.getResources(domainType, domainUuid, aclType);
        if (StringUtils.isNotBlank(aclTemplates)) {
            //Get the ACLEntries...
            return api.parseJsonString(aclType, aclTemplates);
        } else {
            throw new NuageVspException(
                    aclType + " ACL corresponding to network " + networkUuid + " does not exist in VSP. There is a data sync issue. Please a check VSP or create a new network");
        }
    }

    public List<NuageVspObject> getACLEntriesAssociatedToLocation(String aclNetworkLocationId, NuageVspEntity aclTemplateType, String aclTemplateId) throws NuageVspException {
        NuageVspEntity aclEntryType = aclTemplateType.equals(NuageVspEntity.INGRESS_ACLTEMPLATES) ?
                NuageVspEntity.INGRESS_ACLTEMPLATES_ENTRIES :
                NuageVspEntity.EGRESS_ACLTEMPLATES_ENTRIES;
        String aclTemplateEntries;
        if (aclNetworkLocationId != null) {
            aclTemplateEntries = findEntityUsingFilter(aclTemplateType, aclTemplateId, aclEntryType, NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_ID, aclNetworkLocationId);
        } else {
            aclTemplateEntries = findEntityUsingFilter(aclTemplateType, aclTemplateId, aclEntryType, NuageVspAttribute.ACLTEMPLATES_ENTRY_LOCATION_TYPE,
                                                       Acl.AclEntryLocationType.ANY);
        }
        if (StringUtils.isNotBlank(aclTemplateEntries)) {
            //Get the ACLEntries...
            return api.parseJsonString(aclEntryType, aclTemplateEntries);
        } else {
            return Collections.emptyList();
        }
    }

    // Return Pair to know whether it was created or retrieved
    @Override public Pair<Boolean, String> findOrCreateSharedResource(boolean findByExternalId, SharedNetworkType resourceType, final String staticNatNetworkUuid,
            String staticNatVlanGateway, String staticNatVlanNetmask, boolean staticNatVlanUnderlay, String networkName) throws NuageVspException {
        String sharedNetworkJson = findSharedResource(findByExternalId, resourceType, staticNatNetworkUuid, staticNatVlanGateway, staticNatVlanNetmask);
        if (StringUtils.isNotBlank(sharedNetworkJson)) {
            return Pair.of(false, getEntityId(sharedNetworkJson, NuageVspEntity.SHARED_NETWORK));
        }

        String sharedResourceId = createSharedResource(resourceType, null, staticNatNetworkUuid, staticNatVlanGateway, staticNatVlanNetmask, staticNatVlanUnderlay, networkName);
        return Pair.of(true, sharedResourceId);
    }

    public Optional<NuageVspObject> findSharedResourceWithFallback(boolean findByExternalId, SharedNetworkType resourceType, final String staticNatNetworkUuid,
            String staticNatVlanGateway, String staticNatVlanNetmask) throws NuageVspException {

        String vspSharedNetworkJson = findSharedResource(findByExternalId, resourceType, staticNatNetworkUuid, staticNatVlanGateway, staticNatVlanNetmask);

        if (StringUtils.isNotBlank(vspSharedNetworkJson)) {
            final List<NuageVspObject> sharedNetworks = api.parseJsonString(NuageVspEntity.SHARED_NETWORK, vspSharedNetworkJson);

            final Optional<NuageVspObject> firstSharedNetwork = Optional.of(sharedNetworks.get(0));
            if (sharedNetworks.size() == 1) {
                return firstSharedNetwork;
            }

            String cidr = NetUtils.getCidrFromGatewayAndNetmask(staticNatVlanGateway, staticNatVlanNetmask);
            final String externalId = UuidUtils.generateUuidFromCidr(cidr);

            Predicate<NuageVspObject> hasCidrBasedUuid = new Predicate<NuageVspObject>() {
                @Override public boolean apply(NuageVspObject input) {
                    return externalId.equals(input.getExternalId());
                }
            };
            Predicate<NuageVspObject> hasSubnetUuidBasedUuid = new Predicate<NuageVspObject>() {
                @Override public boolean apply(NuageVspObject input) {
                    return StringUtils.equals(staticNatNetworkUuid, input.getExternalId());
                }
            };

            Optional<NuageVspObject> match = Iterables.tryFind(sharedNetworks, hasCidrBasedUuid);
            if (!match.isPresent()) {
                match = Iterables.tryFind(sharedNetworks, hasSubnetUuidBasedUuid);
            }

            return match.or(firstSharedNetwork);
        }

        return Optional.absent();

    }

    public String findSharedResource(boolean findByExternalId, SharedNetworkType resourceType, String staticNatNetworkUuid, String staticNatVlanGateway,
            String staticNatVlanNetmask) throws NuageVspException {
        String address = NetUtils.getSubNet(staticNatVlanGateway, staticNatVlanNetmask);

        NuageVspFilter filter = where().field(NuageVspAttribute.SHARED_RESOURCE_ADRESS)
                                       .eq(address)
                                       .and()
                                       .field(NuageVspAttribute.SHARED_RESOURCE_NETMASK)
                                       .eq(staticNatVlanNetmask)
                                       .and()
                                       .field(NuageVspAttribute.SHARED_RESOURCE_TYPE)
                                       .eq(resourceType);

        if (findByExternalId) {
            filter = filter.and()
                           .field(NuageVspAttribute.EXTERNAL_ID)
                           .eq(staticNatNetworkUuid);
        }

        return findEntityUsingFilter(NuageVspEntity.SHARED_NETWORK, null, null, filter);
    }

    public String createSharedResource(SharedNetworkType resourceType, String sharedResourceParentId, String staticNatNetworkUuid, String staticNatVlanGateway,
            String staticNatVlanNetmask, boolean staticNatVlanUnderlay, String networkName) throws NuageVspException {
        SubnetUtils subnetUtils = new SubnetUtils(staticNatVlanGateway, staticNatVlanNetmask);

        NuageVspObject sharedNtwkEntity = api.createNuageVspObject(NuageVspEntity.SHARED_NETWORK);
        sharedNtwkEntity.set(NuageVspAttribute.NAME, resourceType.name()
                                                                 .charAt(0) + "-" + staticNatNetworkUuid);
        sharedNtwkEntity.set(NuageVspAttribute.DESCRIPTION, networkName);
        sharedNtwkEntity.setExternalId(staticNatNetworkUuid);
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_GATEWAY, staticNatVlanGateway);
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_NETMASK, staticNatVlanNetmask);
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_ADRESS, subnetUtils.getInfo()
                                                                                  .getNetworkAddress());
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_TYPE, resourceType);
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_UNDERLAY, staticNatVlanUnderlay);
        sharedNtwkEntity.set(NuageVspAttribute.SHARED_RESOURCE_PARENT_ID, sharedResourceParentId);

        try {
            sharedNtwkEntity = api.createResource(sharedNtwkEntity);
            String sharedResourceId = sharedNtwkEntity.getId();
            s_logger.debug("Nuage Vsp Shared Network is not available. So, created a new Shared Network " + staticNatNetworkUuid);
            return sharedResourceId;
        } catch (NuageVspApiException e) {
            String errorMessage = "Failed to create SharedResource in VSP using REST API. Json response from VSP REST API is " + e.getMessage();
            s_logger.error(errorMessage);
            throw new NuageVspException(errorMessage);
        }
    }

    @Override public void deleteSharedResourceInVSP(SharedNetworkType resourceType, String staticNatNetworkUuid, String staticNatVlanGateway, String staticNatVlanNetmask)
            throws NuageVspException {
        String sharedResourceJson = findSharedResource(true, resourceType, staticNatNetworkUuid, staticNatVlanGateway, staticNatVlanNetmask);
        if (StringUtils.isNotBlank(sharedResourceJson)) {
            String sharedNetworkId = getEntityId(sharedResourceJson, NuageVspEntity.SHARED_NETWORK);
            api.deleteResource(NuageVspEntity.SHARED_NETWORK, sharedNetworkId);
        }
    }

    @Override public String applyStaticNatInVsp(NetworkDetails attachedNetworkDetails, String vportId, VspStaticNat vspStaticNat) throws NuageVspException {
        String sharedNetworkId = findOrCreateSharedResource(false, SharedNetworkType.FLOATING, vspStaticNat.getVlanUuid(), vspStaticNat.getVlanGateway(),
                                                            vspStaticNat.getVlanNetmask(), vspStaticNat.isVlanUnderlay(), null).getRight();

        if (StringUtils.isNotBlank(vportId)) {
            return allocateFIPToVPortInVsp(attachedNetworkDetails, vportId, sharedNetworkId, vspStaticNat.getIpAddress(), vspStaticNat.getIpUuid(), vspStaticNat.getVspNic()
                                                                                                                                                                .getSecondaryIpAddress(),
                                           vspStaticNat.getVspNic()
                                                       .getSecondaryIpUuid());
        } else {
            if (attachedNetworkDetails.getDomainType()
                                      .equals(NuageVspEntity.DOMAIN)) {
                String vportIdUsingNicUuid = findEntityIdByExternalUuid(NuageVspEntity.SUBNET, attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT, vspStaticNat.getVspNic()
                                                                                                                                                                       .getUuid());
                if (StringUtils.isNotBlank(vportIdUsingNicUuid)) {
                    s_logger.warn("NIC associated to Static NAT " + vspStaticNat.getIpAddress() + "(" + vspStaticNat.getIpUuid() + ") is not present in VSD. But, VM's VPort"
                                          + " with externalID " + vspStaticNat.getVspNic()
                                                                              .getUuid() + " exists in VSD. So, associate the FIP to the Vport " + vportIdUsingNicUuid);
                    return allocateFIPToVPortInVsp(attachedNetworkDetails, vportId, sharedNetworkId, vspStaticNat.getIpAddress(), vspStaticNat.getIpUuid(), vspStaticNat.getVspNic()
                                                                                                                                                                        .getSecondaryIpAddress(),
                                                   vspStaticNat.getVspNic()
                                                               .getSecondaryIpUuid());
                } else {
                    s_logger.warn("Static NAT " + vspStaticNat.getIpAddress() + "(" + vspStaticNat.getIpUuid() + ")" + " is not associated to the VM interface because "
                                          + "neither the interface nor the VPort with NIC's UUID " + vspStaticNat.getVspNic()
                                                                                                                 .getUuid() + " is not present in VSD");
                }
            }
        }
        return null;
    }

    @Override public String allocateFIPToVPortInVsp(NetworkDetails attachedNetworkDetails, String vportId, String vspSharedNetworkId, String staticNatIp, String staticNatIpUuid,
            String nicSecondaryIp4Address, String nicSecondaryIpUuid) throws NuageVspException {
        String fipExternalId = attachedNetworkDetails.getDomainUuid() + ":" + staticNatIpUuid;
        String floatingIpId = null;
        boolean isFipCreated = false;

        try {
            //Check if the floating IP is already associated to the Domain/L2Domain
            floatingIpId = findEntityIdByExternalUuid(attachedNetworkDetails.getDomainType(), attachedNetworkDetails.getDomainId(), NuageVspEntity.FLOATING_IP, fipExternalId);
            if (StringUtils.isBlank(floatingIpId)) {
                NuageVspObject floatingIpEntity = api.createNuageVspObject(NuageVspEntity.FLOATING_IP);
                floatingIpEntity.set(NuageVspAttribute.FLOATING_IP_ADDRESS, staticNatIp);
                floatingIpEntity.set(NuageVspAttribute.ASSOC_SHARED_NTWK_ID, vspSharedNetworkId);
                floatingIpEntity.setExternalId(fipExternalId);
                isFipCreated = true;

                NuageVspObject floatingIpJson = api.createResource(attachedNetworkDetails.getDomainType(), attachedNetworkDetails.getDomainId(), floatingIpEntity);
                floatingIpId = floatingIpJson.getId();
                s_logger.debug("Created a new FloatingIP in Vsp " + floatingIpJson + " in FLoatingIP shared resource " + vspSharedNetworkId);

            }

            //Create a Floating IP with the given IP under the domain
            if (nicSecondaryIp4Address != null) {
                createVirtualIPWithFloatingIPId(vportId, nicSecondaryIp4Address, nicSecondaryIpUuid, floatingIpId);
                s_logger.debug("Associated the new FloatingIP " + staticNatIp + " to Virtual IP " + nicSecondaryIp4Address + " of VM with VPort " + vportId);
            } else {
                updateVPortWithFloatingIPId(vportId, floatingIpId);
                s_logger.debug("Associated the new FloatingIP " + staticNatIp + " to VM with VPort " + vportId);
            }

        } catch (NuageVspApiException e1) {
            if (isFipCreated && !api.deleteQuietly(NuageVspEntity.FLOATING_IP, floatingIpId)) {
                s_logger.warn("Failed to rollback the creation of the FIP");
            }

            throw new NuageVspException("Failed to create Floating in VSP using REST API " + e1.getMessage());
        }

        return floatingIpId;
    }

    private void createVirtualIPWithFloatingIPId(String vportId, String nicSecondaryIp4Address, String nicSecondaryIpUuid, String floatingIPId) throws NuageVspException {
        //set the floating IP to the VPort
        NuageVspObject virtualIp = api.createNuageVspObject(NuageVspEntity.VIRTUAL_IP);
        virtualIp.set(NuageVspAttribute.VIRTUAL_IP_FLOATING_IP_ID, floatingIPId);
        virtualIp.set(NuageVspAttribute.VIRTUAL_IP_ADDRESS, nicSecondaryIp4Address);
        virtualIp.setExternalId(nicSecondaryIpUuid);

        try {
            String virtualIpsJson = findEntityUsingFilter(NuageVspEntity.VPORT, vportId, NuageVspEntity.VIRTUAL_IP, NuageVspAttribute.VIRTUAL_IP_ADDRESS, nicSecondaryIp4Address);
            List<NuageVspObject> virtualIps = api.parseJsonString(NuageVspEntity.VIRTUAL_IP, virtualIpsJson);
            if (CollectionUtils.isNotEmpty(virtualIps)) {
                NuageVspObject existingVirtualIp = Iterables.getFirst(virtualIps, null);
                String existingFloatingIpId = existingVirtualIp.get(NuageVspAttribute.VIRTUAL_IP_FLOATING_IP_ID);
                if (existingFloatingIpId != null && !existingFloatingIpId.equals(floatingIPId)) {
                    throw new NuageVspException(
                            "Failed to associate the FloatingIP " + floatingIPId + " to the VPort " + vportId + ". Another VirtualIP with the same vip = " + nicSecondaryIp4Address
                                    + " exists.");
                } else {
                    return;
                }
            }

            api.createResource(NuageVspEntity.VPORT, vportId, virtualIp);
        } catch (NuageVspException e) {
            if (!e.isNoChangeInEntityException()) {
                throw new NuageVspException("Failed to associate the FloatingIP " + floatingIPId + " to the VPort " + vportId, e);
            }
        }
    }

    @Override public void updateVPortWithFloatingIPId(String vportId, String floatingIPId) throws NuageVspException {
        //set the floating IP to the VPort
        NuageVspObject vportEntity = api.createNuageVspObject(NuageVspEntity.VPORT);
        vportEntity.setId(vportId);
        vportEntity.set(NuageVspAttribute.VPORT_FLOATING_IP_ID, floatingIPId);

        try {
            api.updateResource(vportEntity);
        } catch (NuageVspApiException e) {
            if (!e.isNoChangeInEntityException()) {
                throw NuageVspRestApi.handleException("Failed to associated the FloatingIP %s to the VPort %s", e, floatingIPId, vportId);
            }
        }
    }

    @Override public void releaseFIPFromVsp(NetworkDetails attachedNetworkDetails, String vportId, String staticNatIpUuid, String nicSecondaryIpUuid) throws NuageVspException {
        //get the FIP
        String fipExternalId = attachedNetworkDetails.getDomainUuid() + ":" + staticNatIpUuid;
        final String floatingIpId = findEntityIdByExternalUuid(attachedNetworkDetails.getDomainType(), attachedNetworkDetails.getDomainId(), NuageVspEntity.FLOATING_IP,
                                                               fipExternalId);

        String virtualIpId = null;
        if (nicSecondaryIpUuid != null) {
            virtualIpId = findEntityIdByExternalUuid(attachedNetworkDetails.getSubnetType(), attachedNetworkDetails.getSubnetId(), NuageVspEntity.VIRTUAL_IP, nicSecondaryIpUuid);
        }

        if (StringUtils.isBlank(floatingIpId)) {
            s_logger.debug("vportId is null and also FIP with external ID " + staticNatIpUuid + " does not exists in VSP");
            return;
        }

        if (StringUtils.isBlank(vportId)) {
            s_logger.debug("vportId is null. This could be case where VM interface is not present in VSP. So, finding the VPort in " + attachedNetworkDetails.getSubnetId()
                                   + " that has the FIP with externalId " + staticNatIpUuid);

            String vportJson = api.getResources(NuageVspEntity.FLOATING_IP, floatingIpId, NuageVspEntity.VPORT);
            if (StringUtils.isNotBlank(vportJson)) {
                List<NuageVspObject> vports = api.parseJsonString(NuageVspEntity.VPORT, vportJson);
                vportId = vports.get(0)
                                .get(NuageVspAttribute.ID);
            } else {
                vportJson = api.getResources(attachedNetworkDetails.getSubnetType(), attachedNetworkDetails.getSubnetId(), NuageVspEntity.VPORT);
                List<NuageVspObject> vports = api.parseJsonString(NuageVspEntity.VPORT, vportJson);
                for (NuageVspObject vport : vports) {

                    String virtualIpsJson = api.getResources(NuageVspEntity.VPORT, (String)vport.get(NuageVspAttribute.ID), NuageVspEntity.VIRTUAL_IP);
                    Collection<NuageVspObject> virtualIps = api.parseJsonString(NuageVspEntity.VIRTUAL_IP, virtualIpsJson);
                    virtualIps = Collections2.filter(virtualIps, new Predicate<NuageVspObject>() {
                        @Override public boolean apply(NuageVspObject input) {
                            return input.get(NuageVspAttribute.VIRTUAL_IP_FLOATING_IP_ID)
                                        .equals(floatingIpId);
                        }
                    });

                    if (CollectionUtils.isNotEmpty(virtualIps)) {
                        vportId = vport.get(NuageVspAttribute.ID);
                        virtualIpId = Iterables.getFirst(virtualIps, null)
                                               .get(NuageVspAttribute.ID);
                        s_logger.debug("Found a VPort " + vport + " that is associated the stale FIP " + fipExternalId + " in network " + attachedNetworkDetails.getDomainUuid());
                        break;
                    }
                }
            }
        }

        if (StringUtils.isBlank(vportId)) {
            // No vport, just cleanup Floating IP
            api.deleteQuietly(NuageVspEntity.FLOATING_IP, floatingIpId);
            return;
        }

        if (virtualIpId != null) {
            api.deleteQuietly(NuageVspEntity.VIRTUAL_IP, virtualIpId);
        } else {
            updateVPortWithFloatingIPId(vportId, null);
        }
        s_logger.debug("Removed the association of Floating IP " + fipExternalId + " with VSP VPort " + vportId);

        if (!api.deleteQuietly(NuageVspEntity.FLOATING_IP, floatingIpId)) {
            throw new NuageVspException("Failed to remove Floating IP");
        }

    }

    @Override public String findEntityUsingFilter(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, NuageVspFilter filter) throws NuageVspException {
        try {
            String jsonString;
            if (childEntityType == null && entityId == null) {
                jsonString = api.getResources(entityType, filter);
            } else {
                jsonString = api.getResources(entityType, entityId, childEntityType, filter);
            }

            return jsonString;
        } catch (NuageVspApiException exception) {
            throw NuageVspRestApi.handleException("Failed to execute REST API call to VSP to get %s using VSP filter %s.  Json response from VSP REST API is %s", exception,
                                                  entityType, filter, exception.getMessage());
        }
    }

    private NuageVspObject findVMInterface(List<NuageVspObject> vmInterfaces, String macAddress) throws NuageVspException {
        for (NuageVspObject vmInterface : vmInterfaces) {
            if (vmInterface.hasAttribute(NuageVspAttribute.VM_INTERFACE_MAC) && StringUtils.equals(vmInterface.<String>get(NuageVspAttribute.VM_INTERFACE_MAC), macAddress)) {
                return vmInterface;
            }
        }
        return null;
    }

    @Override public String getEnterpriseName(String domainName, String domainPath) {
        domainPath = StringUtils.removeStart(domainPath, "/");

        if (!StringUtils.contains(domainPath, "/")) {
            return domainName;
        }

        return domainPath.replace("/", "-");
    }

    @Override public String getResources(NuageVspEntity entityType) throws NuageVspException {
        return api.getResources(entityType);
    }

    @Override public <T> T getResources(NuageVspEntity entityType, Class<T> type) throws NuageVspException {
        String json = api.getResources(entityType);
        return api.parseJsonString(entityType, json, type);
    }

    @Override public String findEntityUsingFilter(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, NuageVspAttribute filterAttr, Object filterAttrValue)
            throws NuageVspException {
        return findEntityUsingFilter(entityType, entityId, childEntityType, api.createFilter(filterAttr, filterAttrValue));
    }

    @Override public String findEntityIdByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId) throws NuageVspException {
        final NuageVspObject entity = api.getEntityByExternalId(entityType, entityId, childEntityType, externalId);
        return entity != null ? entity.getId() : null;
    }

    @Override @SuppressWarnings("unchecked") public <T> T findFieldValueByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType,
            String externalId, NuageVspAttribute fieldName) throws NuageVspException {
        final NuageVspObject entity = api.getEntityByExternalId(entityType, entityId, childEntityType, externalId);
        return entity != null ? entity.<T>get(fieldName) : null;
    }

    @Override public String findEntityByExternalUuid(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId) throws NuageVspException {
        try {
            return findEntityUsingFilter(entityType, entityId, childEntityType, NuageVspAttribute.EXTERNAL_ID, externalId);
        } catch (NuageVspApiException exception) {
            throw NuageVspRestApi.handleException("Failed to execute REST API call to VSP to get %s using VSP filter %s. ", exception, entityType, externalId);
        }
    }

    @Override public String getEntityId(String jsonString, NuageVspEntity entityType) throws NuageVspException {
        String id = "";
        if (StringUtils.isNotBlank(jsonString)) {
            List<NuageVspObject> entityDetails = api.parseJsonString(entityType, jsonString);
            id = entityDetails.iterator()
                              .next()
                              .get(NuageVspAttribute.ID);
        }
        return id;
    }

    @Override public void cleanUpDomainAndTemplate(String enterpriseId, String networkUuid, String domainTemplateName) throws NuageVspException {
        //get the L3 DomainTemplate with externalUuid
        String networkDomainTemplateId = findFieldValueByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN, networkUuid, NuageVspAttribute.TEMPLATE_ID);
        String domainTemplateEntity = findEntityUsingFilter(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE, NuageVspAttribute.NAME, domainTemplateName);
        String domainTemplateId = getEntityId(domainTemplateEntity, NuageVspEntity.DOMAIN_TEMPLATE);
        String vspNetworkId;

        // If the VSP Domain was deleted already, this is null
        if (networkDomainTemplateId == null) {
            return;
        }

        if (networkDomainTemplateId.equals(domainTemplateId)) {
            vspNetworkId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN, networkUuid);
            if (StringUtils.isNotBlank(vspNetworkId)) {
                s_logger.debug("Found a VSP L3 network " + vspNetworkId + " that corresponds to network " + networkUuid + " in CS. So, delete it");
                api.deleteQuietly(NuageVspEntity.DOMAIN, vspNetworkId);
            }
        } else {
            vspNetworkId = findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE, networkUuid);
            if (StringUtils.isNotBlank(vspNetworkId)) {
                s_logger.debug("Found a VSP L3 network " + vspNetworkId + " that corresponds to network " + networkUuid + " in CS. So, delete it");
                api.deleteQuietly(NuageVspEntity.DOMAIN_TEMPLATE, vspNetworkId);
            }
        }
    }

    @Override public void createVportInVsp(VspNic vspNic, VspNetwork vspNetwork, List<NuageVspObject> vmInterfaceList, NetworkDetails attachedNetworkDetails,
            DhcpOptions dhcpOptions) throws NuageVspException {

        List<String> vPortIds = new ArrayList<String>();
        for (NuageVspObject vmInterface : vmInterfaceList) {

            String vmInterfaceUuid = vmInterface.getExternalId();
            //Create a VPort for each VM interface. This its association will not be deleted when VM is stopped
            //then set the VPortID in the VM Interface
            //Check if VPort already exists
            NuageVspEntity networkTypeToBeAttached = attachedNetworkDetails.getSubnetType();
            String networkIdToBeAttached = attachedNetworkDetails.getSubnetId();
            String vPortId = findEntityIdByExternalUuid(networkTypeToBeAttached, networkIdToBeAttached, NuageVspEntity.VPORT, vmInterfaceUuid);
            if (StringUtils.isBlank(vPortId)) {
                NuageVspConstants.AddressSpoofing spoofing = vspNic.isRequiresSpoofing() ? NuageVspConstants.AddressSpoofing.ENABLED : NuageVspConstants.AddressSpoofing.INHERITED;
                NuageVspObject vmPortEntity = api.createNuageVspObject(NuageVspEntity.VPORT)
                                                 .set(NuageVspAttribute.NAME, vmInterfaceUuid)
                                                 .set(NuageVspAttribute.DESCRIPTION, vmInterface.get(NuageVspAttribute.VM_INTERFACE_MAC))
                                                 .set(NuageVspAttribute.VPORT_ACTIVE, true)
                                                 .set(NuageVspAttribute.VPORT_TYPE, "VM")
                                                 .set(NuageVspAttribute.VPORT_ADDRESSSPOOFING, spoofing)
                                                 .setExternalId(vmInterfaceUuid);

                try {
                    NuageVspObject vport = api.createResource(networkTypeToBeAttached, networkIdToBeAttached, vmPortEntity);
                    s_logger.debug("Created VPort for network " + networkTypeToBeAttached + " with ID " + networkIdToBeAttached + " in Nuage. Response from VSP is " + vport);
                    vPortId = vport.getId();
                    vPortIds.add(vPortId);

                } catch (NuageVspApiException e) {
                    //clean up all the previously created VPorts
                    for (String vportId : vPortIds) {
                        api.deleteQuietly(NuageVspEntity.VPORT, vportId);
                    }
                    throw NuageVspRestApi.handleException("Failed to create VPort in VSP using REST API. Json response from VSP REST API is %s ", e, e.getMessage());
                }
            }

            if (dhcpOptions != null) {
                for (DhcpOption dhcpOption : dhcpOptions.getOptions()) {
                    NuageVspObject dhcpOptionToCreate = createDhcpOption(vmInterfaceUuid, dhcpOption);
                    api.createResource(NuageVspEntity.VPORT, vPortId, dhcpOptionToCreate);
                }
            }

            if (vspNetwork.isShared() && vspNetwork.isPublicAccess()) {
                try {
                    String nicIp = vmInterface.get(NuageVspAttribute.VM_INTERFACE_IPADDRESS);
                    makeInterfacePubliclyReachable(vspNetwork, nicIp, attachedNetworkDetails, vPortId, vmInterfaceUuid);
                } catch (NuageVspException e) {
                    //clean up all the previously created VPorts
                    for (String vportId : vPortIds) {
                        api.deleteQuietly(NuageVspEntity.VPORT, vportId);
                    }
                    throw e;
                }

            }

            //Set the VPort information in then VMInsterfaceList
            vmInterface.set(NuageVspAttribute.VM_INTERFACE_VPORT_ID, vPortId);
        }
    }

    @Override public String generateCmsIdForNuageVsp(String cmsName) throws NuageVspException {
        try {
            // Get Cloud Management System ID
            NuageVspObject cmsEntity = api.createNuageVspObject(NuageVspEntity.CLOUD_MGMT_SYSTEMS);
            cmsEntity.set(NuageVspAttribute.NAME, cmsName);

            NuageVspObject cms = api.createResource(cmsEntity);
            s_logger.debug("Retrieved CMS ID for VSP . Response from VSP is " + cms);
            return cms.getId();
        } catch (NuageVspApiException e) {
            throw NuageVspRestApi.handleException("Failed to retrieve CMS ID VSP.", e);
        }
    }

    @Override public boolean removeCmsIdForNuageVsp(String cmsId) throws NuageVspException {
        try {
            if (StringUtils.isBlank(cmsId)) {
                throw new NuageVspApiException("CMS Id is null while trying to remove the CMS Id.");
            }

            api.deleteResource(NuageVspEntity.CLOUD_MGMT_SYSTEMS, cmsId);
            s_logger.debug("Deleted CMS ID for VSP.");
            return true;
        } catch (NuageVspApiException exception) {
            throw NuageVspRestApi.handleException("Failed to delete CMS ID VSP.", exception);
        }
    }

    @Override public boolean isKnownCmsIdForNuageVsp(String cmsId) throws NuageVspException {
        try {
            api.getResource(NuageVspEntity.CLOUD_MGMT_SYSTEMS, cmsId, false);
            return true;
        } catch (NuageVspApiException exception) {
            if (exception.getHttpErrorCode() == 404) {
                return false;
            }

            throw NuageVspRestApi.handleException("Failed to retrieve CMS ID VSP.", exception);
        }
    }

    @Override public List<NuageVspObject> parseJsonString(NuageVspEntity entity, String json) throws NuageVspException {
        return api.parseJsonString(entity, json);
    }

    @Override public <T> T parseJsonString(NuageVspEntity entity, String json, Class<T> type) throws NuageVspException {
        return api.parseJsonString(entity, json, type);
    }

    @Override public void deleteQuietly(NuageVspEntity entityType, String entityId) {
        api.deleteQuietly(entityType, entityId);
    }

    @Override public void login() throws NuageVspApiException {
        api.login();
    }

    @Override public String getResources(NuageVspEntity enterprise, NuageVspFilter filter) throws NuageVspApiException {
        return api.getResources(enterprise, filter);
    }

    @Override public String getResources(NuageVspObject parent, NuageVspEntity childEntityType) throws NuageVspApiException {
        return api.getResources(parent, childEntityType);
    }

    @Override
    public NuageVspObject getResource(NuageVspEntity nuageEntityType, String entityId) throws NuageVspApiException {
        return api.getResource(nuageEntityType, entityId);
    }

    @Override
    public boolean isExistingResource(NuageVspEntity entity, String entiyId){
        return api.isExistingResource(entity, entiyId);
    }
}
