/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import net.nuage.vsp.acs.client.api.NuageVspAclClient;
import net.nuage.vsp.acs.client.api.NuageVspApiClient;
import net.nuage.vsp.acs.client.api.NuageVspManagerClient;
import net.nuage.vsp.acs.client.api.model.VspApiDefaults;
import net.nuage.vsp.acs.client.api.model.VspDomain;
import net.nuage.vsp.acs.client.api.model.VspDomainCleanUp;
import net.nuage.vsp.acs.client.api.model.VspDomainTemplate;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.model.NuageVspVersions;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.exception.NuageVspException;

import org.apache.commons.lang.StringUtils;

import static net.nuage.vsp.acs.client.common.model.NuageVspFilter.where;

public class NuageVspManagerClientImpl extends NuageVspClientImpl  implements NuageVspManagerClient {

    public NuageVspManagerClientImpl(NuageVspApiClient nuageVspApiClient, NuageVspAclClient nuageVspAclClient) {
        super(nuageVspApiClient, nuageVspAclClient);
    }

    @Override
    public Pair<Boolean, String> auditNuageVspCmsId(String nuageVspCmsId, boolean auditOnly) throws NuageVspException {
        if (StringUtils.isNotBlank(nuageVspCmsId)) {
            boolean knownCmsId = nuageVspApiClient.isKnownCmsIdForNuageVsp(nuageVspCmsId);
            if (!knownCmsId) {
                return Pair.of(false, nuageVspCmsId);
            }
            return Pair.of(true, nuageVspCmsId);
        } else if (!auditOnly) {
            nuageVspCmsId = registerNuageVspCmsId();
            if (StringUtils.isNotBlank(nuageVspCmsId)) {
                return Pair.of(true, nuageVspCmsId);
            }
        }
        return Pair.of(false, null);
    }

    @Override
    public String registerNuageVspCmsId() throws NuageVspException {
        return nuageVspApiClient.generateCmsIdForNuageVsp("CMS-System");
    }

    @Override
    public boolean unregisterNuageVspCmsId(String nuageVspCmsId) throws NuageVspException {
        return nuageVspApiClient.removeCmsIdForNuageVsp(nuageVspCmsId);
    }

    @Override
    public boolean isSupportedApiVersion(String version) throws NuageVspException {
        NuageVspApiVersion apiVersion = NuageVspApiVersion.fromString(version);
        return apiVersion.isSupported() && isSupportedVersionVSD(apiVersion);
    }

    @Override
    public VspApiDefaults getApiDefaults() throws NuageVspException {
        return new VspApiDefaults.Builder()
                .version(NuageVspApiVersion.CURRENT.toString())
                .retryCount(NuageVspConstants.DEFAULT_API_RETRY_COUNT)
                .retryInterval(NuageVspConstants.DEFAULT_API_RETRY_INTERVAL)
                .build();
    }
    
    @Override
    public boolean syncDomainWithNuageVsp(VspDomain vspDomain, boolean add, boolean remove) throws NuageVspException {
        if (add) {
            addDomain(vspDomain);
        } else if (remove) {
            deleteDomain(vspDomain);
        }
        return true;
    }

    @Override
    public void addDomain(VspDomain vspDomain) throws NuageVspException {
        nuageVspApiClient.getOrCreateVSPEnterprise(vspDomain.getUuid(), vspDomain.getName(), vspDomain.getPath());
    }

    @Override
    public void deleteDomain(VspDomain vspDomain) throws NuageVspException {
        nuageVspApiClient.deleteEnterpriseInVsp(vspDomain.getUuid(), nuageVspApiClient.getEnterpriseName(vspDomain.getName(), vspDomain.getPath()));
    }

    @Override
    public boolean cleanUpDomain(VspDomainCleanUp vspDomainCleanUp) throws NuageVspException {
        if (vspDomainCleanUp.getSharedNetworkUuids() != null) {
            String enterpriseId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, vspDomainCleanUp.getUuid());
            for (Map.Entry<String, List<String>> entry : vspDomainCleanUp.getSharedNetworkUuids().entrySet()) {
                for (String sharedNetworkUuid : entry.getValue()) {
                    nuageVspApiClient.cleanUpDomainAndTemplate(enterpriseId, sharedNetworkUuid, entry.getKey());
                }
            }
        }

        return true;
    }

    @Override
    public List<VspDomainTemplate> getDomainTemplates(VspDomain vspDomain, String name) throws NuageVspException {
        NuageVspFilter filter = name != null ? where().field(NuageVspAttribute.NAME).contains(name) : null;
        String enterpriseId = nuageVspApiClient.findEntityIdByExternalUuid(NuageVspEntity.ENTERPRISE, null, null, vspDomain.getUuid());
        String json = nuageVspApiClient.findEntityUsingFilter(NuageVspEntity.ENTERPRISE, enterpriseId, NuageVspEntity.DOMAIN_TEMPLATE, filter); //returns json string
        List<NuageVspObject> domainTemplates = nuageVspApiClient.parseJsonString(NuageVspEntity.DOMAIN_TEMPLATE, json);
        List<VspDomainTemplate> result = new LinkedList<>();
        for (NuageVspObject domainTemplate : domainTemplates) {
            if (StringUtils.isBlank(domainTemplate.getExternalId())) {
                result.add(new VspDomainTemplate(domainTemplate.getName(), domainTemplate.getDescription()));
            }
        }

        return result;
    }

    @Override
    public List<NuageVspApiVersion> getSupportedVersionList() throws NuageVspException {
        NuageVspVersions versions = nuageVspApiClient.getResources(NuageVspEntity.VERSION, NuageVspVersions.class);
        List<NuageVspApiVersion> result = new LinkedList<>();

        for (NuageVspObject version : versions.getVersions()) {
            NuageVspApiVersion.Status status = version.get(NuageVspAttribute.VERSION_STATUS);
            if (status != NuageVspApiVersion.Status.UNSUPPORTED) {
                NuageVspApiVersion versionObj = NuageVspApiVersion.fromString(version.<String>get(NuageVspAttribute.VERSION));
                result.add(versionObj);
            }
        }

        return result;
    }

    //Used get SupportedVersionList, rename after remove getSupportedVersions()
    @Deprecated
    @Override
    public Map<NuageVspApiVersion, NuageVspApiVersion.Status> getSupportedVersions() throws NuageVspException {
        NuageVspVersions versions = nuageVspApiClient.getResources(NuageVspEntity.VERSION, NuageVspVersions.class);
        Map<NuageVspApiVersion, NuageVspApiVersion.Status> result = new HashMap<>();

        for (NuageVspObject version : versions.getVersions()) {
            NuageVspApiVersion.Status status = version.get(NuageVspAttribute.VERSION_STATUS);
            if (status != NuageVspApiVersion.Status.UNSUPPORTED) {
                NuageVspApiVersion versionObj = NuageVspApiVersion.fromString(version.<String>get(NuageVspAttribute.VERSION));
                result.put(versionObj, status);
            }

        }

        return result;
    }

    private boolean isSupportedVersionVSD(NuageVspApiVersion version) throws NuageVspException {
        return getSupportedVersionList().contains(version);
    }
}
