/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;

import net.nuage.vsp.acs.client.api.NuageVspStatistics;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpVersion;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.PoolingClientConnectionManager;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.params.HttpProtocolParams;
import org.codehaus.jackson.map.ObjectMapper;
import com.google.common.base.MoreObjects;
import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;

import net.nuage.vsp.acs.client.api.model.NuageVspPaging;
import net.nuage.vsp.acs.client.api.model.NuageVspUser;
import net.nuage.vsp.acs.client.api.model.Protocol;
import net.nuage.vsp.acs.client.api.model.VspHost;
import net.nuage.vsp.acs.client.common.FilterProcessor;
import net.nuage.vsp.acs.client.common.NuageVspApiVersion;
import net.nuage.vsp.acs.client.common.NuageVspConstants;
import net.nuage.vsp.acs.client.common.NuageVspConstants.ErrorCode;
import net.nuage.vsp.acs.client.common.RequestType;
import net.nuage.vsp.acs.client.common.model.Acl;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import net.nuage.vsp.acs.client.common.model.NuageVspError;
import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import net.nuage.vsp.acs.client.common.model.NuageVspObject;
import net.nuage.vsp.acs.client.common.model.Pair;
import net.nuage.vsp.acs.client.common.model.gson.CodedEnumTypeAdapter;
import net.nuage.vsp.acs.client.common.model.gson.IntCodedEnumTypeAdapter;
import net.nuage.vsp.acs.client.common.model.gson.NuageVspObjectTypeAdapter;
import net.nuage.vsp.acs.client.common.model.gson.ProtocolTypeAdapter;
import net.nuage.vsp.acs.client.common.ssl.TLSSocketFactory;
import net.nuage.vsp.acs.client.common.utils.Logger;
import net.nuage.vsp.acs.client.exception.NuageJsonParsingException;
import net.nuage.vsp.acs.client.exception.NuageVspApiException;
import net.nuage.vsp.acs.client.exception.NuageVspAuthenticationException;
import net.nuage.vsp.acs.client.exception.NuageVspConnectivityException;
import net.nuage.vsp.acs.client.exception.NuageVspException;
import net.nuage.vsp.acs.client.exception.UnsupportedNuageEntityException;

import static net.nuage.vsp.acs.client.api.model.NuageVspPaging.paging;
import static net.nuage.vsp.acs.client.common.model.NuageVspFilter.where;

public class NuageVspRestApi {
    private static final Logger s_logger = new Logger(NuageVspRestApi.class);

    private static final long DELAY_FACTOR = 2;
    private volatile Optional<String> apiKey = Optional.absent();
    private NuageVspStatistics statistics;

    protected VspHost vspHost;

    private PoolingClientConnectionManager s_httpClientManager = null;

    private HttpClient client;

    private ObjectMapper mapper = new ObjectMapper();

    public NuageVspRestApi(VspHost vspHost, NuageVspStatistics nuageVspStatistics) {
        this.vspHost = vspHost;
        this.statistics = nuageVspStatistics;
    }

    public NuageVspObject createResource(NuageVspObject entity, NuageVspUser proxyUser) throws NuageVspApiException {
        String json = executeRestApiWithRetry(RequestType.CREATE, entity.getEntityType(), null, null, entity, true, null, null, proxyUser);
        return getFirstJsonEntity(entity.getEntityType(), json);
    }

    public NuageVspObject createResource(NuageVspEntity parentEntityType, String parentEntityId, NuageVspObject childEntity, Boolean checkWarning, NuageVspUser proxyUser,
            ErrorCode... retryNuageErrorCodes) throws NuageVspApiException {
        EnumSet<ErrorCode> retryNuageErrorCodesSet = null;
        if (retryNuageErrorCodes.length != 0) {
            retryNuageErrorCodesSet = EnumSet.copyOf(Arrays.asList(retryNuageErrorCodes));
        }

        String json = executeRestApiWithRetry(RequestType.CREATE, parentEntityType, parentEntityId, childEntity.getEntityType(), childEntity, checkWarning, retryNuageErrorCodesSet,
                                              null, proxyUser);
        return getFirstJsonEntity(childEntity.getEntityType(), json);
    }

    public NuageVspObject createResource(NuageVspObject entity) throws NuageVspApiException {
        return createResource(entity, null);
    }

    public NuageVspObject createResource(NuageVspObject entity, String proxyUserUuid, String proxyUserDomainUuid) throws NuageVspApiException {
        return createResource(entity, new NuageVspUser(proxyUserDomainUuid, proxyUserUuid));
    }

    public NuageVspObject createResource(NuageVspEntity parentEntityType, String parentEntityId, NuageVspObject childEntity) throws NuageVspApiException {
        return createResource(parentEntityType, parentEntityId, childEntity, null, null);
    }

    public NuageVspObject createResource(NuageVspEntity parentEntityType, String parentEntityId, NuageVspObject childEntity, String proxyUserUuid, String proxyUserDomainUuid)
            throws NuageVspApiException {
        return createResource(parentEntityType, parentEntityId, childEntity, null, new NuageVspUser(proxyUserDomainUuid, proxyUserUuid));
    }

    public NuageVspObject createResource(NuageVspObject parent, NuageVspObject child, ErrorCode... retryNuageErrorCodes) throws NuageVspApiException {
        return createResource(parent.getEntityType(), parent.getId(), child, null, null, retryNuageErrorCodes);
    }

    //------------------------------------------------------------------------

    public String updateResource(NuageVspObject nuageVspObject) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.MODIFY, nuageVspObject.getEntityType(), nuageVspObject.getId(), null, nuageVspObject, false, null, null, null);
    }

    public String updateResource(NuageVspEntity parentEntityType, String parentEntityId, NuageVspEntity childEntityType, Object childEntity) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.MODIFYRELATED, parentEntityType, parentEntityId, childEntityType, childEntity, false, null, null, null);
    }

    //------------------------------------------------------------------------

    public String deleteResource(NuageVspEntity entityType, String entityId) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.DELETE, entityType, entityId, null, null, null, null, null, null);
    }

    public String deleteResource(NuageVspObject nuageVspObject) throws NuageVspApiException {
        return deleteResource(nuageVspObject.getEntityType(), nuageVspObject.getId());
    }

    public boolean deleteQuietly(NuageVspObject entityToBeCleaned) {
        try {
            deleteResource(entityToBeCleaned);
            s_logger.debug("Successfully cleaned stale VSP entity %s with ID %s", entityToBeCleaned.getEntityType(), entityToBeCleaned.getId());
            return true;
        } catch (NuageVspApiException e) {
            s_logger.warn("Failed to clean %s with ID %s from NuageVsp. Please contact Nuage Vsp csproot to clean stale objects", entityToBeCleaned.getEntityType(),
                          entityToBeCleaned.getId());
        }
        return false;
    }

    public boolean deleteQuietly(NuageVspEntity entityToBeCleaned, String entityIDToBeCleaned) {
        try {
            deleteResource(entityToBeCleaned, entityIDToBeCleaned);
            s_logger.debug("Successfully cleaned stale VSP entity %s with ID %s", entityToBeCleaned, entityIDToBeCleaned);
            return true;
        } catch (NuageVspApiException e) {
            s_logger.warn("Failed to clean %s with ID %s from NuageVsp. Please contact Nuage Vsp csproot to clean stale objects", entityToBeCleaned, entityIDToBeCleaned);
        }
        return false;
    }

    //------------------------------------------------------------------------

    public String getResources(NuageVspEntity nuageEntityType, NuageVspPaging paging) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.GETALL, nuageEntityType, null, null, null, true, null, paging, null);
    }

    public String getChildResources(NuageVspEntity parentEntityType, String parentEntityId, NuageVspEntity nuageChildEntityType, NuageVspPaging paging)
            throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.GETRELATED, parentEntityType, parentEntityId, nuageChildEntityType, null, true, null, paging, null);
    }

    public String getResource(NuageVspEntity nuageEntityType, String entityId, Boolean checkWarning) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.GET, nuageEntityType, entityId, null, null, checkWarning, null, null, null);
    }

    public String resourceExists(NuageVspEntity nuageEntityType, String entityId, Boolean checkWarning) throws NuageVspApiException {
        return executeRestApiWithRetry(RequestType.HEAD, nuageEntityType, entityId, null, null, checkWarning, null, null, null);
    }

    public NuageVspObject getResource(NuageVspEntity nuageEntityType, String entityId) throws NuageVspApiException {
        String json = getResource(nuageEntityType, entityId, true);
        return getOnlyJsonEntity(nuageEntityType, json);
    }

    public String getResources(NuageVspEntity nuageEntityType, NuageVspFilter filter) throws NuageVspApiException {
        return getResources(nuageEntityType, paging().filter(filter));
    }

    public String getResources(NuageVspEntity nuageEntityType) throws NuageVspApiException {
        return getResources(nuageEntityType, (NuageVspPaging)null);
    }

    public String getResources(NuageVspObject nuageVspObject, NuageVspEntity nuageChildEntityType) throws NuageVspApiException {
        return getChildResources(nuageVspObject.getEntityType(), nuageVspObject.getId(), nuageChildEntityType, null);
    }

    public String getResources(NuageVspObject nuageVspObject, NuageVspEntity nuageChildEntityType, NuageVspFilter filter) throws NuageVspApiException {
        return getChildResources(nuageVspObject.getEntityType(), nuageVspObject.getId(), nuageChildEntityType, paging().filter(filter));
    }

    public String getResources(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType) throws NuageVspApiException {
        return getChildResources(nuageEntityType, entityId, nuageChildEntityType, null);
    }

    public String getResources(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, NuageVspFilter filter) throws NuageVspApiException {
        return getChildResources(nuageEntityType, entityId, nuageChildEntityType, paging().filter(filter));
    }

    public String getResources(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, NuageVspAttribute filterAttribute, Object filterValue)
            throws NuageVspApiException {
        return getChildResources(nuageEntityType, entityId, nuageChildEntityType, paging().filter(createFilter(filterAttribute, filterValue)));
    }

    //------------------------------------------------------------------------

    private Set<String> getLinkedEntities(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType) throws NuageVspApiException {
        String currentRelatedEntitiesJson = getResources(nuageEntityType, entityId, nuageChildEntityType);
        List<NuageVspObject> currentRelatedEntities = parseJsonString(nuageChildEntityType, currentRelatedEntitiesJson);

        Set<String> currentChildEntityIds = new TreeSet<String>();
        for (NuageVspObject currentRelatedEntity : currentRelatedEntities) {
            currentChildEntityIds.add(currentRelatedEntity.getId());
        }

        return currentChildEntityIds;
    }

    protected boolean saveLinkedEntities(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, Set<String> oldNuageChildEntityIds,
            Set<String> newNuageChildEntityIds) throws NuageVspApiException {

        boolean isModified = !oldNuageChildEntityIds.equals(newNuageChildEntityIds);

        if (isModified) {
            updateResource(nuageEntityType, entityId, nuageChildEntityType, newNuageChildEntityIds);
        }

        return isModified;
    }

    public void linkRelatedEntity(NuageVspObject nuageVspObject, NuageVspEntity nuageChildEntityType, String nuageChildEntityId) throws NuageVspException {
        linkRelatedEntity(nuageVspObject.getEntityType(), nuageVspObject.getId(), nuageChildEntityType, nuageChildEntityId);
    }

    public boolean linkRelatedEntity(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, String nuageChildEntityId) throws NuageVspException {

        Set<String> currentChildEntityIds = getLinkedEntities(nuageEntityType, entityId, nuageChildEntityType);
        Set<String> nuageChildEntityIds = new TreeSet<>(currentChildEntityIds);
        nuageChildEntityIds.add(nuageChildEntityId);

        return saveLinkedEntities(nuageEntityType, entityId, nuageChildEntityType, currentChildEntityIds, nuageChildEntityIds);
    }

    public boolean unlinkRelatedEntity(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, String nuageChildEntityId) throws NuageVspException {

        Set<String> currentChildEntityIds = getLinkedEntities(nuageEntityType, entityId, nuageChildEntityType);
        Set<String> nuageChildEntityIds = new TreeSet<>(currentChildEntityIds);
        nuageChildEntityIds.remove(nuageChildEntityId);

        return saveLinkedEntities(nuageEntityType, entityId, nuageChildEntityType, currentChildEntityIds, nuageChildEntityIds);
    }

    public boolean setRelatedEntities(NuageVspObject nuageVspObject, NuageVspEntity nuageChildEntityType, Set<String> nuageChildEntityIds) throws NuageVspException {
        return setRelatedEntities(nuageVspObject.getEntityType(), nuageVspObject.getId(), nuageChildEntityType, nuageChildEntityIds);
    }

    public boolean setRelatedEntities(NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, Set<String> nuageChildEntityIds)
            throws NuageVspException {

        Set<String> currentChildEntityIds = getLinkedEntities(nuageEntityType, entityId, nuageChildEntityType);
        return saveLinkedEntities(nuageEntityType, entityId, nuageChildEntityType, currentChildEntityIds, nuageChildEntityIds);
    }

    protected static NuageVspFilter createFilter(NuageVspAttribute filterAttr, Object filterAttrValue) {
        NuageVspFilter.BinaryOperatorBuilder field = where().field(filterAttr);
        if (filterAttr == NuageVspAttribute.EXTERNAL_ID && filterAttrValue != null) {
            return field.startsWith((String)filterAttrValue);
        } else {
            return field.eq(filterAttrValue);
        }
    }

    public NuageVspObject getOnlyJsonEntity(NuageVspEntity entityType, String jsonString) throws NuageVspApiException {
        return Iterables.getOnlyElement(parseJsonString(entityType, jsonString));
    }

    public NuageVspObject getFirstJsonEntity(NuageVspEntity entityType, String jsonString) throws NuageVspApiException {
        return Iterables.getFirst(parseJsonString(entityType, jsonString), null);
    }

    public NuageVspObject getEntityByExternalId(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, String externalId) throws NuageVspApiException {
        if (externalId == null) {
            return null;
        }

        final NuageVspFilter filter = createFilter(NuageVspAttribute.EXTERNAL_ID, externalId);
        String jsonString;

        if (childEntityType == null && entityId == null) {
            jsonString = getResources(entityType, filter);
        } else {
            jsonString = getResources(entityType, entityId, childEntityType, filter);
        }

        NuageVspEntity entity = MoreObjects.firstNonNull(childEntityType, entityType);
        return getFirstJsonEntity(entity, jsonString);
    }

    public String getEntityId(NuageVspEntity entityType, String jsonString) throws NuageVspApiException {
        String id = "";
        if (StringUtils.isNotBlank(jsonString)) {
            id = getOnlyJsonEntity(entityType, jsonString).getId();
        }
        return id;
    }

    /**
     * API executes the REST API call and returns the response
     *  @param type
     * @param nuageEntityType
     * @param entityId
     * @param nuageChildEntityType
     * @param entityDetails
     * @param checkWarning     TODO    @return
     * @param paging
     * @throws Exception
     */
    private String executeRestApiWithRetry(RequestType type, NuageVspEntity nuageEntityType, String entityId, NuageVspEntity nuageChildEntityType, Object entityDetails,
            Boolean checkWarning, Set<ErrorCode> retryNuageErrorCodes, NuageVspPaging paging, NuageVspUser proxyUser) throws NuageVspApiException {

        int attempt = 1;
        long sleepTime = vspHost.getRetryInterval();
        NuageVspApiException exception = null;
        do {
            Pair<String, String> urlAndResponse = Pair.of("", "");
            try {
                urlAndResponse = executeNuageApi(nuageEntityType, entityId, nuageChildEntityType, entityDetails, type, checkWarning, paging, proxyUser);

                if (attempt > 1) {
                    s_logger.trace("After %s attempt, exception %s was handled and method %s was successfully executed ", --attempt, exception.getMessage(),
                                   "executeHttpRequestWithRetry");
                }
                return urlAndResponse.getRight();
            } catch (NuageVspApiException e) {
                exception = e;
                if (attempt >= 1) {
                    if (attempt <= vspHost.getNoofRetry()) {
                        if (!handleException(attempt, sleepTime, e, "executeHttpRequestWithRetry", urlAndResponse.getLeft(), retryNuageErrorCodes)) {
                            throw e;
                        }
                    }

                    attempt++;
                    sleepTime *= DELAY_FACTOR;
                }
            }
        } while (attempt <= vspHost.getNoofRetry() + 1);

        s_logger.error(String.format("Failed to execute %s method even after %s attempts, due to exception %s ", "executeHttpRequestWithRetry", vspHost.getNoofRetry(),
                                     exception.getMessage()));
        throw exception;
    }

    public Pair<String, String> executeNuageApi(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, Object entityDetails, RequestType type,
            boolean checkWarning, NuageVspPaging paging, NuageVspUser proxyUser, String restRelativePath) {
        return null;
    }

    /**
     * All Nuage APIs will be executed as CMS User in Nuage. This API uses proxy user API to view all the object
     * that are permitted for the user in enterprise. Password of the CMS user will reset when the API fails with
     * Authentication error. This is taken care by retry logic. Look at {@link #login} for
     * more details
     * @param entityType
     * @param entityId
     * @param childEntityType
     * @param entityDetails
     * @param type
     * @param checkWarning     TODO    @return List that contains Map of attributes and their corresponding values
     * @throws Exception
     */
    public Pair<String, String> executeNuageApi(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, Object entityDetails, RequestType type,
            Boolean checkWarning, NuageVspPaging paging, NuageVspUser proxyUser) throws NuageVspApiException {

        statistics.reportVSDCall(MoreObjects.firstNonNull(childEntityType, entityType), type);
        HttpRequestBase httpMethod = null;
        HttpResponse response = null;

        String restUrl = getRestPath(entityType, entityId, childEntityType, type, checkWarning);

        try {
            FilterProcessor.processFilter(paging, vspHost.getNuageVspCmsId());

            URI uri = new URI(restUrl);

            switch (type) {
                case CREATE: {
                    httpMethod = new HttpPost(uri);
                    break;
                }
                case MODIFY:
                case MODIFYRELATED: {
                    httpMethod = new HttpPut(uri);
                    break;
                }
                case DELETE: {
                    httpMethod = new HttpDelete(uri);
                    break;
                }
                case HEAD: {
                    httpMethod = new HttpHead(uri);
                }
                default: // GET, GETALL
                    httpMethod = new HttpGet(uri);
            }

            if (!apiKey.isPresent() && entityType != NuageVspEntity.ME) {
                login();
            }

            setHttpHeaders(httpMethod, entityType, paging, proxyUser);

            if (httpMethod instanceof HttpEntityEnclosingRequestBase) {
                String jsonString;

                if (entityDetails instanceof List || entityDetails instanceof Set || entityDetails instanceof Map || entityDetails instanceof NuageVspObject) {
                    jsonString = getJsonString(entityDetails);
                } else {
                    jsonString = "";
                }

                httpMethod.setHeader("Content-Type", "application/json");
                StringEntity entity = new StringEntity(jsonString);
                ((HttpEntityEnclosingRequestBase)httpMethod).setEntity(entity);
            }

            long time = System.currentTimeMillis();
            if (client == null) {
                //Re-initialize the HTTP client
                createHttpClient();
                s_logger.debug("Http client is not initialized. So, it is re-initialized to execute https %s with url %s.", httpMethod, null);
            }

            response = client.execute(httpMethod);
            s_logger.trace("Total time taken to execute HTTPS method %s %s - %d ms.", type, null, (System.currentTimeMillis() - time));

            String entity = parseHttpResponse(response, childEntityType != null ? childEntityType : entityType, type);
            return Pair.of(restUrl, entity);
        } /*catch (ConnectException e) { //remove again.
            throw new NuageVspConnectivityException("Failed to connect to VSD device.");
        }*/ catch (NuageVspApiException e) {
            throw e;
        } catch (SSLPeerUnverifiedException e) {
            client = null;

            // Specific catch for CLOUD-736 : After upgrade of VSD SSL certificate no longer recognized, reinitialize HTTP client
            createHttpClient();
            s_logger.debug("Reinitialized HTTP client because of new SSL certificate");
            throw new NuageVspApiException(500, null, "Throwing exception to trigger retry with reinitialized HTTP client", e);
        } catch (SSLException | ClientProtocolException e) {
            throw new NuageVspConnectivityException("Failed to setup SSL", e);
        } catch (UnsupportedEncodingException e) {
            throw new NuageJsonParsingException("Unsupported Encoding", e);
        } catch (IOException ioe) {
            s_logger.error("Error while executing HTTPS method " + type + " " + null, ioe);
            throw new NuageVspConnectivityException("Error while executing HTTPS method " + type + " " + null, ioe);
        } catch (URISyntaxException urise) {
            s_logger.error("Error while building URI " + null, urise);
            throw new NuageVspApiException("Error while building URI " + null, urise);
        } finally {
            if (httpMethod != null) {
                httpMethod.releaseConnection();
            }
        }
    }

    protected void createHttpClient() {
        HttpParams params = new BasicHttpParams();
        HttpProtocolParams.setVersion(params, HttpVersion.HTTP_1_1);
        HttpConnectionParams.setSoTimeout(params, 120000);
        HttpConnectionParams.setTcpNoDelay(params, true);

        if (s_httpClientManager == null) {
            s_httpClientManager = new PoolingClientConnectionManager(TLSSocketFactory.createSchemeRegistry());
            s_httpClientManager.setDefaultMaxPerRoute(200);
            s_httpClientManager.setMaxTotal(200);
        }

        client = new DefaultHttpClient(s_httpClientManager, params);
    }

    private void setHttpHeaders(HttpRequestBase httpMethod, NuageVspEntity entityType, NuageVspPaging paging, NuageVspUser proxyUser) {

        //set the authorization header
        NuageVspUser cmsUserInfo = vspHost.getCmsUserInfo();
        StringBuilder authStr = new StringBuilder().append(cmsUserInfo.getUserName())
                                                   .append(":");
        //set the CMS user password only if it is a ME API else set the API even for CMS USer
        if (entityType == NuageVspEntity.ME) {
            authStr.append(cmsUserInfo.getPassword());
        } else {
            authStr.append(apiKey.or(""));
        }
        String encoding = Base64.encodeBase64String(authStr.toString()
                                                           .getBytes());
        httpMethod.setHeader("Authorization", "Basic " + encoding);
        httpMethod.setHeader("X-Nuage-Organization", cmsUserInfo.getEnterpriseName());

        if (httpMethod.getMethod()
                      .equals(""))

        {
            if (proxyUser != null) {
                String proxyUserHeader = proxyUser.getUserName()
                                                  .replace("-", "") + "@" + proxyUser.getEnterpriseName();
                httpMethod.setHeader("X-Nuage-ProxyUser", proxyUserHeader);
            }
        }

        if (paging != null) {
            if (paging.hasFilter()) {
                NuageVspFilter filter = paging.getFilter();
                httpMethod.setHeader("X-Nuage-Filter", filter.getFilterString(vspHost.getApiVersion()));
                if (filter.hasOrderBy()) {
                    httpMethod.setHeader("X-Nuage-OrderBy", filter.getOrderByString());
                }
            }

            if (paging.hasPage()) {
                httpMethod.setHeader("X-Nuage-Page", String.valueOf(paging.getPage()));
            }
            if (paging.hasPageSize()) {
                httpMethod.setHeader("X-Nuage-PageSize", String.valueOf(paging.getPageSize()));
            }
        }
    }

    private String getRestPath(NuageVspEntity entity) {
        StringBuilder url = new StringBuilder(vspHost.getRestRelativePath());
        url.append("/")
           .append(entity.getEntityType());
        return url.toString();
    }

    private String getRestPath(NuageVspEntity entityType, String entityId, NuageVspEntity childEntityType, RequestType type, Boolean checkWarning) {
        assert (childEntityType != null) == type.requiresIdAndChildEntityType();
        StringBuilder url = new StringBuilder();

        if (entityType != NuageVspEntity.VERSION) {
            url.append(vspHost.getRestRelativePath());
        } else {
            url.append(vspHost.getRootPath());
        }

        url.append("/")
           .append(entityType.getEntityType());

        if (childEntityType != null) {
            url.append("/")
               .append(entityId)
               .append("/")
               .append(childEntityType.getEntityType());
        } else if (type.requiresId() && entityType != NuageVspEntity.ME) {
            url.append("/")
               .append(entityId);
        }

        if (checkWarning == null) {
            checkWarning = type.isIdempotent();
        }

        if (!checkWarning) {
            url.append("?responseChoice=1");
        }

        return url.toString();
    }

    /**
     * Login will set the APIKey value for the CMS User. When any REST API call fails with AuthenticationException
     * a retry will call login and set the correct APIKey. There is not retry for this API.
     */
    public void login() throws NuageVspApiException {
        try {
            final Pair<String, String> result = executeNuageApi(NuageVspEntity.ME, null, null, null, RequestType.GET, null, null, null);
            String meJson = result.getRight();
            NuageVspObject me = Iterables.getFirst(parseJsonString(NuageVspEntity.ME, meJson), null);
            if (me == null) {
                apiKey = Optional.absent();
                return;
            } else {
                apiKey = Optional.fromNullable(me.<String>get(NuageVspAttribute.APIKEY));
            }

            String systemEnterpriseId = me.get(NuageVspAttribute.ENTERPRISE_ID);
            if (StringUtils.isNotBlank(systemEnterpriseId)) {
                String groupJson = getResources(NuageVspEntity.ENTERPRISE, systemEnterpriseId, NuageVspEntity.GROUP, where(NuageVspAttribute.GROUP_ROLE).eq("CMS"));
                NuageVspObject cmsGroup = Iterables.getOnlyElement(parseJsonString(NuageVspEntity.GROUP, groupJson), null);
                if (cmsGroup != null) {
                    String cmsUsersJson = getResources(NuageVspEntity.GROUP, (String)cmsGroup.get(NuageVspAttribute.ID), NuageVspEntity.USER);
                    List<? extends NuageVspObject> cmsUsers = parseJsonString(NuageVspEntity.USER, cmsUsersJson);
                    for (NuageVspObject cmsUser : cmsUsers) {
                        String cmsUsername = cmsUser.get(NuageVspAttribute.USER_USERNAME);
                        if (cmsUsername.equals(vspHost.getCmsUserLogin())) {
                            return;
                        }
                    }
                }
            }
        } catch (NuageVspAuthenticationException e) {
            s_logger.warn("Failed to authenticate on Nuage VSP device. Provided credentials are invalid.", e);
            throw e;
        } catch (NuageVspException e) {
            throw handleException("Failed to check if the user is part of the CMS group", e);
        }

        s_logger.error("User '" + vspHost.getCmsUserLogin() + "' is not part of the CMS group!");
        throw new NuageVspApiException("User '" + vspHost.getCmsUserLogin() + "' is not part of the CMS group!");
    }

    public String getJsonString(Object entityDetails) throws NuageJsonParsingException {
        try {
            return getGsonInstance(null).toJson(entityDetails);

        } catch (JsonSyntaxException jsonException) {
            s_logger.error("Error while retrieving JSON from complex data", jsonException);
            throw new NuageJsonParsingException("Error while retrieving JSON from complex data", jsonException);
        }
    }

    /**
     * API parses the HTTPRespose for HTTP errors. If the HTTP status is success then it parses the reponse JSON string and returns
     * List that contains Map of attributes and their corresponding values
     */
    private String parseHttpResponse(HttpResponse httpResponse, NuageVspEntity entityType, RequestType type) throws NuageVspApiException {
        String jsonResult = getJsonFromResponse(httpResponse);

        final int statusCode = httpResponse.getStatusLine()
                                           .getStatusCode();

        String errorMessage = "NUAGE HTTP REQUEST %s: HTTP Response code: %d: Response : %s";

        if (statusCode >= 402 && statusCode <= 599) {
            NuageVspError error;
            if (httpResponse.getFirstHeader("Content-Type")
                            .getValue()
                            .equals("application/json")) {
                error = parseJsonError(jsonResult);
                if (statusCode == 404) {
                    error.setInternalErrorCode(ErrorCode.RESOURCE_NOT_FOUND);
                }
            }
             else {
                error = new NuageVspError(ErrorCode.RESOURCE_NOT_FOUND, null, null);
            }

            ErrorCode nuageErrorCode = error.getInternalErrorCode();
            String nuageErrorDetails = error.getStackTrace();

            if (nuageErrorCode != null || nuageErrorDetails != null) {
                //TODO: This is a temporary hack to avoid printing internal error code 2039
                if (nuageErrorCode == NuageVspConstants.ErrorCode.DUPLICATE_ACL_PRIORITY) {
                    s_logger.warn(errorMessage, "FAILED", statusCode, jsonResult);
                } else if (type == RequestType.DELETE && nuageErrorCode == NuageVspConstants.ErrorCode.RESOURCE_NOT_FOUND) {
                    // be gentle on deleting already deleted objects
                    s_logger.info(errorMessage, "FAILED", statusCode, jsonResult);
                    return jsonResult;
                } else if (nuageErrorCode != NuageVspConstants.ErrorCode.NO_CHANGE_IN_ENTITY) {
                    s_logger.error(errorMessage, "FAILED", statusCode, jsonResult);
                }
                throw new NuageVspApiException(statusCode, errorMessage, nuageErrorCode, nuageErrorDetails, entityType, type);
            }
        } else if (statusCode == 400) {
            s_logger.trace(errorMessage, statusCode, jsonResult);
            throw new NuageVspApiException(httpResponse.getStatusLine()
                                                       .getStatusCode(), errorMessage, null, "The request sent by the client was syntactically incorrect", entityType, type);
        } else if (statusCode == 401) {
            s_logger.trace(errorMessage);
            throw new NuageVspAuthenticationException(errorMessage);
        } else if (statusCode >= 200 && statusCode <= 299) {
            s_logger.trace(errorMessage, "RESULT", statusCode, jsonResult);
        }
        return jsonResult;
    }

    private static String getJsonFromResponse(HttpResponse httpResponse) throws NuageJsonParsingException {
        HttpEntity entity = httpResponse.getEntity();

        if (entity != null) {
            try (InputStream inputStream = entity.getContent()) {
                return IOUtils.toString(inputStream);
            } catch (IOException ioe) {
                throw new NuageJsonParsingException(null, ioe);
            }
        } else {
            return "";
        }
    }

    private NuageVspError parseJsonError(String jsonResult) throws NuageJsonParsingException {

        NuageVspError error = new GsonBuilder().registerTypeAdapter(ErrorCode.class, new IntCodedEnumTypeAdapter())
                                               .create()
                                               .fromJson(jsonResult, NuageVspError.class);
        return error;
    }

    public List<NuageVspObject> parseJsonString(NuageVspEntity nuageEntity, String jsonResult) throws NuageJsonParsingException, UnsupportedNuageEntityException {
        if (StringUtils.isBlank(jsonResult)) {
            return new LinkedList<>();
        }
        return parseJsonString(nuageEntity, jsonResult, NuageVspObjectTypeAdapter.LIST_TYPE);
    }

    public <T> T parseJsonString(NuageVspEntity nuageEntity, String jsonResult, Type type) throws NuageJsonParsingException, UnsupportedNuageEntityException {
        final NuageVspApiVersion apiVersion = vspHost.getApiVersion();
        Collection<String> attributeNames = nuageEntity.getAttributeNameList(apiVersion);

        if (attributeNames == null) {
            throw new UnsupportedNuageEntityException(nuageEntity.getEntityType() + " is not defined in NuageEntity enum. Please add it if it needs to be supported");
        }

        Gson gson = getGsonInstance(nuageEntity);

        try {
            return gson.fromJson(jsonResult, type);
        } catch (JsonSyntaxException ioe) {
            throw new NuageJsonParsingException("Failed to parse the Json response from VSP REST API. Json string is " + jsonResult, ioe);
        }
    }

    private Gson getGsonInstance(NuageVspEntity nuageEntity) {
        return new GsonBuilder().registerTypeAdapter(NuageVspObject.class, new NuageVspObjectTypeAdapter(nuageEntity, vspHost.getApiVersion(), vspHost.getNuageVspCmsId()))
                                .registerTypeAdapter(Protocol.class, new ProtocolTypeAdapter())
                                .registerTypeAdapter(Acl.AclEtherType.class, new CodedEnumTypeAdapter())
                                .serializeNulls()
                                .create();
    }

    protected NuageVspObject createNuageVspObject(NuageVspEntity entity) {
        return new NuageVspObject(vspHost.getApiVersion(), entity);
    }

    protected NuageVspObject createNuageVspObject(NuageVspEntity entity, String entityId) {
        return new NuageVspObject(vspHost.getApiVersion(), entity, entityId);
    }

    protected static NuageVspApiException handleException(String message, NuageVspException cause, Object... args) {
        if (cause instanceof NuageVspApiException) {
            return (NuageVspApiException)cause;
        } else {
            if (args.length > 0) {
                message = String.format(message, args);
            }
            s_logger.error(message, cause);
            return new NuageVspApiException(message, cause);
        }
    }

    private boolean handleException(int attempt, long sleepTime, Exception e, String methodName, String url, Set<ErrorCode> retryNuageErrorCodes) throws NuageVspApiException {
        boolean retry = false;
        Throwable rootCause = ExceptionUtils.getRootCause(e);
        rootCause = rootCause == null ? e : rootCause;

        if (rootCause instanceof NuageVspAuthenticationException) {
            s_logger.trace(String.format("Authentication failed so failed to execute Nuage VSP API %s", url));
            printRetryMessage(attempt, sleepTime, methodName, url, rootCause);
            try {
                retry = true;
                Thread.sleep(sleepTime);
                login();
            } catch (InterruptedException e1) {
                s_logger.warn("Retry sleeping got interrupted");
                throw new NuageVspApiException("Retry sleeping got interrupted", e);
            } catch (NuageVspException nve) {
                s_logger.error("Failed to login to VSD", nve);
                throw new NuageVspApiException("Failed to login to VSD", nve);
            }
        } else if (rootCause instanceof SSLException || e instanceof SSLException) {
            printRetryMessage(attempt, sleepTime, methodName, url, rootCause);
            retry = setRetryFlag(sleepTime);
        } else if (rootCause instanceof NuageVspApiException) {
            NuageVspApiException exception = (NuageVspApiException)rootCause;
            if (exception.getHttpErrorCode() == 500 || (retryNuageErrorCodes != null && retryNuageErrorCodes.contains(exception.getNuageErrorCode()))) {
                printRetryMessage(attempt, sleepTime, methodName, url, rootCause);
                retry = setRetryFlag(sleepTime);
            }
        }
        return retry;
    }

    private static void printRetryMessage(int attempt, long sleepTime, String methodName, String url, Throwable rootCause) {
        s_logger.trace("Failed to execute Nuage VSP API %s : %s", url, rootCause.getMessage());
        s_logger.trace("Attempt %s to re-execute the method %s", attempt, methodName);
        s_logger.trace("Waiting %s millis before re-executing the method %s", sleepTime, methodName);
    }

    private static boolean setRetryFlag(long sleepTime) throws NuageVspApiException {
        try {
            Thread.sleep(sleepTime);
        } catch (InterruptedException e) {
            Thread.currentThread()
                  .interrupt();
            s_logger.warn("Retry sleeping got interrupted");
            throw new NuageVspApiException("Retry sleeping got interrupted", e);
        }
        return true;
    }

    public boolean isExistingResource(NuageVspEntity entity, String entityId) {
        try {
            return resourceExists(entity, entityId, true) != null;
        } catch (NuageVspException e) {
            return false;
        }
    }
}
