/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.impl;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import net.nuage.vsp.acs.client.common.RequestType;
import net.nuage.vsp.acs.client.common.model.NuageVspEntity;
import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.lang3.mutable.MutableLong;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

/**
 * Created by sgoeminn on 1/24/17.
 */
public class NuageVspStatisticsImpl implements net.nuage.vsp.acs.client.api.NuageVspStatistics {
    private AtomicLong globalCount = new AtomicLong();


    private Table<NuageVspEntity, RequestType, AtomicLong> entityCount = HashBasedTable.create(
            NuageVspEntity.values().length,
            RequestType.values().length);

    @Override
    public void reportVSDCall(NuageVspEntity entity, RequestType requestType) {
        synchronized (entityCount) {
            AtomicLong atomicLong = entityCount.get(entity, requestType);
            if (atomicLong == null) {
                atomicLong = new AtomicLong(0L);
                entityCount.put(entity, requestType, atomicLong);
            }
            atomicLong.incrementAndGet();
        }
        globalCount.incrementAndGet();
    }

    @Override
    public Map<NuageVspEntity, Map<RequestType, AtomicLong>> getVsdCountReport() {
        return entityCount.rowMap();
    }

    @Override
    public long getVsdCount(NuageVspEntity entity, RequestType type) {
        AtomicLong atomicLong = entityCount.get(entity, type);
        return atomicLong != null ? atomicLong.get() : 0;
    }

    @Override
    public long getVsdCount(NuageVspEntity entity) {
        long count = entityCount.row(entity)
                .values()
                .stream()
                .mapToLong(AtomicLong::get)
                .sum();
        return count;
    }

    @Override
    public long getVsdCount(RequestType type) {
        long count = entityCount.column(type)
                .values()
                .stream()
                .mapToLong(AtomicLong::get)
                .sum();
        return count;
    }

    @Override
    public long getVsdCount() {
        return globalCount.get();
    }
}
