/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.model;

import net.nuage.vsp.acs.client.common.NuageVspApiVersion;

public class VspHost extends VspObject {

    private final String restRelativePath;
    private final String rootPath;
    private final NuageVspUser cmsUser;
    private final int noofRetry;
    private final long retryInterval;
    private final String nuageVspCmsId;
    private final NuageVspApiVersion apiVersion;

    private VspHost(NuageVspApiVersion apiVersion, String restRelativePath, String rootPath, NuageVspUser cmsUser, int noofRetry, long retryInterval, String nuageVspCmsId) {
        this.apiVersion = apiVersion;
        this.restRelativePath = restRelativePath;
        this.rootPath = rootPath;
        this.cmsUser = cmsUser;
        this.noofRetry = noofRetry;
        this.retryInterval = retryInterval;
        this.nuageVspCmsId = nuageVspCmsId;
    }

    public NuageVspApiVersion getApiVersion() {
        return apiVersion;
    }

    public String getRestRelativePath() {
        return restRelativePath;
    }

    public String getRootPath(){
        return rootPath;
    }

    public String getCmsUserEnterprise() {
        return cmsUser.getEnterpriseName();
    }

    public String getCmsUserLogin() {
        return cmsUser.getUserName();
    }

    public String getCmsUserPassword() {
        return cmsUser.getPassword();
    }

    public NuageVspUser getCmsUserInfo() {
        return cmsUser;
    }

    public int getNoofRetry() {
        return noofRetry;
    }

    public long getRetryInterval() {
        return retryInterval;
    }

    public String getNuageVspCmsId() {
        return nuageVspCmsId;
    }

    public static class Builder extends VspBuilder<Builder, VspHost> {
        private String restRelativePath ;
        private String rootPath;
        private String cmsUserEnterprise;
        private String cmsUserLogin;
        private String cmsUserPassword;
        private NuageVspUser cmsUser;
        private int noofRetry;
        private long retryInterval;
        private String nuageVspCmsId;
        private NuageVspApiVersion apiVersion;

        public Builder apiVersion(NuageVspApiVersion apiVersion) {
            this.apiVersion = apiVersion;
            return this;
        }

        public Builder restRelativePath(String restRelativePath) {
            this.restRelativePath = restRelativePath;
            return this;
        }

        public Builder rootPath(String rootPath){
            this.rootPath = rootPath;
            return this;
        }

        public Builder cmsUser(NuageVspUser cmsUser) {
            this.cmsUser = cmsUser;
            return this;
        }

        public Builder cmsUserEnterprise(String cmsUserEnterprise) {
            this.cmsUserEnterprise = cmsUserEnterprise;
            return this;
        }

        public Builder cmsUserLogin(String cmsUserLogin) {
            this.cmsUserLogin = cmsUserLogin;
            return this;
        }

        public Builder cmsUserPassword(String cmsUserPassword) {
            this.cmsUserPassword = cmsUserPassword;
            return this;
        }

        public Builder noofRetry(int noofRetry) {
            this.noofRetry = noofRetry;
            return this;
        }

        public Builder retryInterval(long retryInterval) {
            this.retryInterval = retryInterval;
            return this;
        }

        public Builder nuageVspCmsId(String nuageVspCmsId) {
            this.nuageVspCmsId =  nuageVspCmsId;
            return this;
        }

        @Override
        public VspHost build() {
            if(cmsUser == null) {
                cmsUser = new NuageVspUser(cmsUserEnterprise, cmsUserLogin, cmsUserPassword);
            }
            return new VspHost(apiVersion, restRelativePath, rootPath, cmsUser, noofRetry,
                    retryInterval, nuageVspCmsId);
        }

        @Override
        public Builder fromObject(VspHost obj) {
            return new Builder().apiVersion(obj.getApiVersion()).restRelativePath(obj.getRestRelativePath()).cmsUserEnterprise(obj.getCmsUserEnterprise())
                    .cmsUserLogin(obj.getCmsUserLogin()).cmsUserPassword(obj.getCmsUserPassword()).noofRetry(obj.getNoofRetry())
                    .retryInterval(obj.getRetryInterval()).nuageVspCmsId(obj.getNuageVspCmsId());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof VspHost)) return false;

        VspHost vspHost = (VspHost) o;

        if (noofRetry != vspHost.noofRetry) return false;
        if (retryInterval != vspHost.retryInterval) return false;
        if (cmsUser != null ? !cmsUser.equals(vspHost.cmsUser) : vspHost.cmsUser != null)
            return false;
        if (nuageVspCmsId != null ? !nuageVspCmsId.equals(vspHost.nuageVspCmsId) : vspHost.nuageVspCmsId != null)
            return false;
        if (restRelativePath != null ? !restRelativePath.equals(vspHost.restRelativePath) : vspHost.restRelativePath != null)
            return false;
        if(rootPath != null ? !rootPath.equals(vspHost.rootPath) : vspHost.rootPath != null)
            return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = restRelativePath != null ? restRelativePath.hashCode() : 0;
        result = 31 * result + (cmsUser != null ? cmsUser.hashCode() : 0);
        result = 31 * result + noofRetry;
        result = 31 * result + (int) (retryInterval ^ (retryInterval >>> 32));
        result = 31 * result + (nuageVspCmsId != null ? nuageVspCmsId.hashCode() : 0);
        return result;
    }
}