/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.api.model;

import java.util.Objects;

public class VspNic extends VspObject {
    private final String uuid;
    private final String macAddress;
    private final Boolean useStaticIp;
    private boolean requiresSpoofing;

    private final String ip;
    private final String secondaryIpAddress;
    private final String secondaryIpUuid;

    private VspNic(String uuid, String macAddress, Boolean useStaticIp, String ip, String secondaryIpAddress, String secondaryIpUuid, boolean requiresSpoofing) {
        this.uuid = uuid;
        this.macAddress = macAddress;
        this.useStaticIp = useStaticIp;
        this.ip = ip;
        this.secondaryIpAddress = secondaryIpAddress;
        this.secondaryIpUuid = secondaryIpUuid;
        this.requiresSpoofing = requiresSpoofing;
    }

    public String getUuid() {
        return uuid;
    }

    public String getMacAddress() {
        return macAddress;
    }

    public Boolean getUseStaticIp() {
        return useStaticIp;
    }

    public boolean isRequiresSpoofing() {
        return requiresSpoofing;
    }

    public String getIp() {
        return ip;
    }

    public String getSecondaryIpAddress() {
        return secondaryIpAddress;
    }

    public String getSecondaryIpUuid() {
        return secondaryIpUuid;
    }

    public static class Builder extends VspBuilder<Builder, VspNic> {
        private String uuid;
        private String macAddress;
        private boolean requiresSpoofing;
        private Boolean useStaticIp;
        private String ip;
        private String secondaryIpAddress;
        private String secondaryIpUuid;

        public Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public Builder macAddress(String macAddress) {
            this.macAddress = macAddress;
            return this;
        }

        public Builder useStaticIp(Boolean useStaticIp) {
            this.useStaticIp = useStaticIp;
            return this;
        }

        public Builder ip(String ip) {
            this.ip = ip;
            return this;
        }

        public Builder requiresSpoofing() {
            this.requiresSpoofing = true;
            return this;
        }

        public Builder withRequiresSpoofing(boolean requiresSpoofing) {
            this.requiresSpoofing = requiresSpoofing;
            return this;
        }

        public Builder secondaryIpAddress(String secondaryIpAddress) {
            this.secondaryIpAddress = secondaryIpAddress;
            return this;
        }

        public Builder secondaryIpUuid(String secondaryIpUuid) {
            this.secondaryIpUuid = secondaryIpUuid;
            return this;
        }

        @Override
        public VspNic build() {
            return new VspNic(uuid, macAddress, useStaticIp, ip, secondaryIpAddress, secondaryIpUuid, requiresSpoofing);
        }

        @Override
        public Builder fromObject(VspNic obj) {
            return new Builder()
                    .uuid(obj.getUuid())
                    .macAddress(obj.getMacAddress())
                    .useStaticIp(obj.getUseStaticIp())
                    .withRequiresSpoofing(obj.isRequiresSpoofing())
                    .ip(obj.getIp())
                    .secondaryIpAddress(obj.getSecondaryIpAddress())
                    .secondaryIpUuid(obj.getSecondaryIpUuid());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof VspNic)) return false;

        VspNic other = (VspNic) o;

        return Objects.equals(this.ip, other.ip)
                && Objects.equals(this.macAddress, other.macAddress)
                && Objects.equals(this.secondaryIpAddress, other.secondaryIpAddress)
                && Objects.equals(this.secondaryIpUuid, other.secondaryIpUuid)
                && Objects.equals(this.useStaticIp, other.useStaticIp)
                && Objects.equals(this.uuid, other.uuid)
                && Objects.equals(this.requiresSpoofing, other.requiresSpoofing);
    }

    @Override
    public int hashCode() {
        int result = uuid != null ? uuid.hashCode() : 0;
        result = 31 * result + (macAddress != null ? macAddress.hashCode() : 0);
        result = 31 * result + (useStaticIp != null ? useStaticIp.hashCode() : 0);
        result = 31 * result + (ip != null ? ip.hashCode() : 0);
        result = 31 * result + (secondaryIpAddress != null ? secondaryIpAddress.hashCode() : 0);
        result = 31 * result + (secondaryIpUuid != null ? secondaryIpUuid.hashCode() : 0);
        return result;
    }
}
