/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common;

import java.util.EnumSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.nuage.vsp.acs.client.api.model.NuageVspPaging;
import net.nuage.vsp.acs.client.common.model.NuageVspAttribute;

import net.nuage.vsp.acs.client.common.model.NuageVspFilter;
import org.apache.commons.lang.StringUtils;

/**
 * This class is used to process the filter for externalId fields conversion from [objectUuid] to [nuageVspCmsId_objectUuid]
 */
public class FilterProcessor {
    public static final Set<NuageVspAttribute> EXTERNAL_ID_FIELDS = EnumSet.of(NuageVspAttribute.EXTERNAL_ID);

    public static void processFilter(NuageVspPaging paging, String nuageVspCmsId) {
        if (paging != null && paging.hasFilter()) {
            paging.filter(processFilter(paging.getFilter(), nuageVspCmsId));
        }
    }

    public static NuageVspFilter processFilter(NuageVspFilter filter, String nuageVspCmsId) {
        if (filter != null && nuageVspCmsId != null) {
            filter.applyCmsId(nuageVspCmsId);
        }

        return filter;
    }

    public static String processFilter(String filter, String nuageVspCmsId) {
        if (filter == null) return null;
        if (nuageVspCmsId == null) return filter;

        for (NuageVspAttribute externalIdField : EXTERNAL_ID_FIELDS) {
            if (!filter.contains(externalIdField.getAttributeName())) continue;

            String regex = ".*?" + externalIdField + ".*?['\"](.*?)['\"]( (?i)and ['\"](.*?)['\"])?";
            Pattern pattern = Pattern.compile(regex);
            Matcher matcher = pattern.matcher(filter);
            StringBuilder newFilter = new StringBuilder();
            StringBuilder oldFilter = new StringBuilder();
            while (matcher.find()) {
                String output = replaceOldExternalId(matcher.group(0), matcher.group(1), nuageVspCmsId);

                if (matcher.group(3) != null) {
                    output = replaceOldExternalId(output, matcher.group(3), nuageVspCmsId);
                }
                newFilter.append(output);
                oldFilter.append(matcher.group(0));
            }
            filter = newFilter.toString() + filter.substring(oldFilter.length());
        }

        return filter;
    }

    private static String replaceOldExternalId(String input, String oldExternalId, String nuageVspCmsId) {
        if (StringUtils.isNotBlank(oldExternalId) && !oldExternalId.endsWith(nuageVspCmsId)) {
            String newExternalId = oldExternalId + NuageVspConstants.EXTERNAL_ID_DELIMITER + nuageVspCmsId;
            return input.replaceAll(oldExternalId, newExternalId);
        }
        return input;
    }
}
