/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.apache.commons.lang.StringUtils;

/**
 * @author waegemae
 */
public class DhcpOptions {

    public static final String CLEAR_IP = "0.0.0.0";
    public static final String CLEAR = "\u0000";
    public static final String CLEAR_DNS = "00";
    public static final int GATEWAY = 3;
    public static final int DNS_SERVERS = 6;
    public static final int HOST_NAME = 12;
    public static final int DOMAIN_NAME = 15;

    List<DhcpOption> options = Lists.newLinkedList();
    Set<Integer> toBeRemoved = new HashSet<>();

    public DhcpOptions() {

    }

    /**
     * Create DHCP options to be defined on subnet level
     * @param vrIp
     * @param dnsServers
     * @param networkDomain
     */
    public DhcpOptions(String vrIp, List<String> dnsServers, String networkDomain) {
        // Make sure the VR is the first in the list
        ArrayList<String> allsDnsServers = vrIp == null ? Lists.<String>newArrayList() : Lists.newArrayList(vrIp);
        if (dnsServers != null) {
            allsDnsServers.addAll(dnsServers);
        }
        if (!allsDnsServers.isEmpty()) {
            options.add(new DhcpOption(DNS_SERVERS, allsDnsServers));
        }
        if (StringUtils.isNotBlank(networkDomain)) {
            options.add(new DhcpOption(DOMAIN_NAME, networkDomain));
        }
    }

    /**
     * Create DHCP options to be defined on VM interface level
     * @param defaultHasDns
     * @param hostname
     * @param networkHasDns
     * @param defaultInterface
     * @param domainRouter
     */
    public DhcpOptions(boolean defaultHasDns, String hostname, boolean networkHasDns, boolean defaultInterface, boolean domainRouter) {
        /*
          - if the network has DNS
            - Set hostname
          - if interface is not the default
            - Clear Gateway, to make sure default route is not assigned
            - Clear (external) Dns servers, if either the network or the default has DNS, to make sure requests are sent to the internal DNS
            - Clear Domain name, if the network has DNS, to make sure only the domain name from the default interface is taken.
         */

        if (!domainRouter) {
            if (defaultHasDns) {
                if (defaultInterface) {
                    options.add(new DhcpOption(HOST_NAME, hostname));
                } else {
                    options.add(new DhcpOption(GATEWAY, CLEAR_IP));
                    options.add(new DhcpOption(DNS_SERVERS, CLEAR_DNS));
                    if (networkHasDns) {
                        options.add(new DhcpOption(DOMAIN_NAME, CLEAR));
                        options.add(new DhcpOption(HOST_NAME, hostname));
                    }
                }
            } else {
                if (!defaultInterface) {
                    options.add(new DhcpOption(GATEWAY, CLEAR_IP));
                    if (networkHasDns) {
                        options.add(new DhcpOption(DNS_SERVERS, CLEAR_DNS));   // overrule the subnet option back to 0
                        options.add(new DhcpOption(DOMAIN_NAME, CLEAR));   // overrule the subnet option back to 0
                        options.add(new DhcpOption(HOST_NAME, hostname));
                    }
                }
            }
        }

        /*
        if (!domainRouter) {
            if (networkHasDns) {
                options.add(new DhcpOption(HOST_NAME, hostname));
            }

            if (!defaultInterface) {
                options.add(new DhcpOption(GATEWAY, CLEAR_IP));
                if (networkHasDns) {
                    options.add(new DhcpOption(DOMAIN_NAME, CLEAR_IP));
                }
                if (defaultHasDns || networkHasDns) {
                    options.add(new DhcpOption(DNS_SERVERS, CLEAR_IP));
                }
            }
        }*/
    }

    public List<DhcpOption> getOptions() {
        return options;
    }

    public void addOption(DhcpOption option) {
        options.add(option);
    }

    public void addOptions(Iterable<DhcpOption> options) {
        Iterables.addAll(this.options, options);
    }

    public void addOptionToRemove(Integer optionCode) { toBeRemoved.add(optionCode); }

    public Set<Integer> getOptionsToBeRemoved() { return toBeRemoved; }
}
