/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.ssl;

import java.io.IOException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.HashSet;
import java.util.Set;

import javax.net.ssl.SSLSocket;

import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLInitializationException;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.conn.ssl.X509HostnameVerifier;

public class TLSSocketFactory extends SSLSocketFactory {
    public static final TrustStrategy TRUST_ALL_CERTIFICATES_STRATEGY = new TrustStrategy() {
        @Override
        public boolean isTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            return true;
        }
    };

    public TLSSocketFactory(TrustStrategy trustStategy, X509HostnameVerifier hostNameVerifier) throws KeyManagementException, UnrecoverableKeyException,
            NoSuchAlgorithmException, KeyStoreException {
        super(trustStategy, hostNameVerifier);
    }

    @Override
    protected void prepareSocket(final SSLSocket socket) throws IOException {
        // Strip "SSLv3" from the current enabled protocols.
        String[] protocols = socket.getSupportedProtocols();
        Set<String> set = new HashSet<String>();
        for (String s : protocols) {
            if (s.equals("SSLv3") || s.equals("SSLv2Hello")) {
                continue;
            }
            set.add(s);
        }
        socket.setEnabledProtocols(set.toArray(new String[0]));
    }

    public static TLSSocketFactory getSocketFactory() {
        try {
            return new TLSSocketFactory(TRUST_ALL_CERTIFICATES_STRATEGY, SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
        } catch (java.security.GeneralSecurityException ex) {
            throw new SSLInitializationException("Failure initializing default SSL context", ex);
        }
    }

    /**
     * Initializes default scheme registry based on JSSE defaults. System properties will
     * not be taken into consideration.
     */
    public static SchemeRegistry createSchemeRegistry() {
        SchemeRegistry registry = new SchemeRegistry();
        registry.register(
                new Scheme("http", 80, PlainSocketFactory.getSocketFactory()));
        registry.register(
                new Scheme("https", 443, TLSSocketFactory.getSocketFactory()));
        return registry;
    }
}
