/**
 * Copyright © 2016 Nokia
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.nuage.vsp.acs.client.common.utils;

import java.nio.ByteBuffer;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;

import org.apache.commons.net.util.SubnetUtils;

public class UuidUtils {
    private static final class CIDR implements Comparable<CIDR>{
        Long[] cidrAsLong;
        String lowAddress;
        String highAddress;
        SubnetUtils.SubnetInfo subnet;
        CIDR (String cidrString) {
            cidrAsLong = NetUtils.cidrToLong(cidrString);
            if (cidrAsLong[1] == 0L) {
                lowAddress = "0.0.0.0";
                highAddress = "255.255.255.255";
            } else {
                subnet = new SubnetUtils(cidrString).getInfo();
                lowAddress = subnet.getLowAddress();
                highAddress = subnet.getHighAddress();
            }
        }

        @Override
        public int compareTo(CIDR o) {
            int result = lowAddress.compareTo(o.lowAddress);
            if (result != 0 ) return result;
            return o.highAddress.compareTo(highAddress);
        }
    }

    public static String generateUuidFromExternalIdAndIp(String externalId, String ip) {
        byte[] externalIdBytes = externalId.getBytes();
        byte[] ipBytes = ip.getBytes();

        ByteBuffer bb = ByteBuffer.wrap(new byte[externalIdBytes.length + ipBytes.length]);
        bb.put(externalIdBytes);
        bb.put(ipBytes);

        UUID uuid = UUID.nameUUIDFromBytes(bb.array());
        return uuid.toString();
    }

    /**
     * Generates a
     * @param string
     * @param base
     * @return
     */
    public static String generateUuidFromString(String string, String base) {
        byte[] bytes = string.getBytes();

        if (base != null) {
            int size = bytes.length + 16;
            ByteBuffer bb = ByteBuffer.wrap(new byte[size]);

            UUID namespace = UUID.fromString(base);
            bb.putLong(namespace.getMostSignificantBits());
            bb.putLong(namespace.getLeastSignificantBits());
            bb.put(bytes);

            bytes = bb.array();
        }

        UUID uuid = UUID.nameUUIDFromBytes(bytes);
        return uuid.toString();
    }

    public static String generateUuidFromCidr(String cidrList) {
        return generateUuidFromCidr(cidrList, null);
    }

    public static String generateUuidFromCidr(String cidrList, String base) {
        Set<CIDR> cidrs = new TreeSet<CIDR>();
        for (String cidr : cidrList.split(",\\s*")) {
            cidrs.add(new CIDR(cidr));
        }

        int size = cidrs.size() * 12;

        if (base != null) {
            size += 16;
        }

        ByteBuffer bb = ByteBuffer.wrap(new byte[size]);

        if (base != null) {
            UUID namespace = UUID.fromString(base);
            bb.putLong(namespace.getMostSignificantBits());
            bb.putLong(namespace.getLeastSignificantBits());
        }

        for (CIDR cidr : cidrs) {
            bb.putLong(cidr.cidrAsLong[0]);
            bb.putInt(cidr.cidrAsLong[1].intValue());
        }

        UUID uuid = UUID.nameUUIDFromBytes(bb.array());
        return uuid.toString();
    }
}
